<?php
namespace TotalTheme;

use TotalTheme\Footer\Core as Footer;
use WP_Query;

defined( 'ABSPATH' ) || exit;

/**
 * Footer Builder.
 *
 * @package Total WordPress theme
 * @subpackage Theme Builder
 * @version 5.7.0
 *
 * @todo move out to Footer_Builder or rename to Theme_Builder\Footer
 */
final class FooterBuilder {

	/**
	 * Start things up.
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Hook into actions and filters.
	 */
	public function init_hooks() {
		if ( wpex_is_request( 'admin' ) ) {
			add_action( 'admin_menu', array( $this, 'add_admin_submenu_page' ), 20 );
			add_action( 'admin_init', array( $this, 'register_page_options' ) );
			add_action( 'wp_ajax_wpex_footer_builder_edit_links', array( $this, 'ajax_edit_links' ) );
		}

		if ( $builder_post_id = self::get_template_id()
			|| ! empty( $_GET[ 'wpex_inline_footer_template_editor' ] )
		) {

			if ( ! get_theme_mod( 'footer_builder_footer_widgets', false ) ) {
				add_filter( 'wpex_register_footer_sidebars', '__return_false' );
			}

			if ( wpex_is_request( 'admin' ) ) {
				add_filter( 'wpex_customizer_panels', array( $this, 'remove_customizer_panels' ) );
				add_filter( 'wpex_customizer_sections', array( $this, 'alter_customizer_settings' ) );
				add_filter( 'wpex_typography_settings', array( $this, 'remove_typography_settings' ) );
			}

			if ( wpex_is_request( 'frontend' ) ) {
				add_action( 'wp', array( $this, 'alter_footer' ) );
				add_filter( 'wpex_vc_css_ids', array( $this, 'wpex_vc_css_ids' ) );
				add_filter( 'wpex_head_css', array( $this, 'wpex_head_css' ), 99 );

			}

			if ( wpex_vc_is_inline() ) {
				add_filter( 'template_include', array( $this, 'builder_template' ), 9999 );
			}

		}
	}

	/**
	 * Returns footer template ID.
	 */
	public static function get_template_id() {
		$id = get_theme_mod( 'footer_builder_page_id' );

		/**
		 * Filters the footer builder template id.
		 *
		 * @param int $id
		 */
		$id = (int) apply_filters( 'wpex_footer_builder_page_id', $id );

		if ( ! empty( $id ) && is_numeric( $id ) ) {
			$id = wpex_parse_obj_id( $id, 'page' ) ?: $id;
			if ( 'publish' == get_post_status( $id ) ) {
				return $id;
			}
		}
	}

	/**
	 * Add sub menu page.
	 */
	public function add_admin_submenu_page() {
		add_submenu_page(
			WPEX_THEME_PANEL_SLUG,
			esc_html__( 'Footer Builder', 'total' ),
			esc_html__( 'Footer Builder', 'total' ),
			'edit_theme_options',
			WPEX_THEME_PANEL_SLUG . '-footer-builder',
			array( $this, 'create_admin_page' )
		);
	}

	/**
	 * Function that will register admin page options.
	 */
	public function register_page_options() {

		// Register settings
		register_setting(
			'wpex_footer_builder',
			'footer_builder',
			array(
				'sanitize_callback' => array( $this, 'save_options' ),
				'default' => null
			)
		);

		// Add main section to our options page
		add_settings_section(
			'wpex_footer_builder_main',
			false,
			array( $this, 'section_main_callback' ),
			'wpex-footer-builder-admin'
		);

		// Custom Page ID
		add_settings_field(
			'footer_builder_page_id',
			esc_html__( 'Footer Builder page', 'total' ),
			array( $this, 'content_id_field_callback' ),
			'wpex-footer-builder-admin',
			'wpex_footer_builder_main',
			array(
				'label_for' => 'wpex-footer-builder-select',
			)
		);

		// Footer Bottom
		add_settings_field(
			'footer_builder_footer_bottom',
			esc_html__( 'Footer Bottom', 'total' ),
			array( $this, 'footer_builder_footer_bottom_field_callback' ),
			'wpex-footer-builder-admin',
			'wpex_footer_builder_main',
			array(
				'label_for' => 'wpex-footer-builder-footer-bottom',
			)
		);

		// Footer Widgets
		add_settings_field(
			'footer_builder_footer_widgets',
			esc_html__( 'Footer Widgets', 'total' ),
			array( $this, 'footer_widgets_field_callback' ),
			'wpex-footer-builder-admin',
			'wpex_footer_builder_main',
			array(
				'label_for' => 'wpex-footer-builder-widgets',
			)
		);

		// Fixed Footer
		add_settings_field(
			'fixed_footer',
			esc_html__( 'Fixed Footer', 'total' ),
			array( $this, 'fixed_footer_field_callback' ),
			'wpex-footer-builder-admin',
			'wpex_footer_builder_main',
			array(
				'label_for' => 'wpex-footer-builder-fixed',
			)
		);

		// Footer Reveal
		add_settings_field(
			'footer_reveal',
			esc_html__( 'Footer Reveal', 'total' ),
			array( $this, 'footer_reveal_field_callback' ),
			'wpex-footer-builder-admin',
			'wpex_footer_builder_main',
			array(
				'label_for' => 'wpex-footer-builder-reveal',
			)
		);

		// Bg
		add_settings_field(
			'bg',
			esc_html__( 'Background Color', 'total' ),
			array( $this, 'bg_field_callback' ),
			'wpex-footer-builder-admin',
			'wpex_footer_builder_main',
			array(
				'label_for' => 'wpex-footer-builder-bg',
			)
		);

		// BG img
		add_settings_field(
			'bg_img',
			esc_html__( 'Background Image', 'total' ),
			array( $this, 'bg_img_field_callback' ),
			'wpex-footer-builder-admin',
			'wpex_footer_builder_main',
			array(
				'label_for' => 'wpex-footer-builder-bg-img',
			)
		);

		// BG img style
		add_settings_field(
			'bg_img_style',
			esc_html__( 'Background Image Style', 'total' ),
			array( $this, 'bg_img_style_field_callback' ),
			'wpex-footer-builder-admin',
			'wpex_footer_builder_main',
			array(
				'label_for' => 'wpex-footer-builder-bg-style',
			)
		);

	}

	/**
	 * Save options.
	 */
	public function save_options( $options ) {

		if ( empty( $options ) ) {
			return;
		}

		// Update footer builder page ID
		if ( ! empty( $options['content_id'] ) ) {
			set_theme_mod( 'footer_builder_page_id', $options['content_id'] );
		} else {
			remove_theme_mod( 'footer_builder_page_id' );
		}

		// Footer Bottom - Disabled by default
		if ( empty( $options['footer_builder_footer_bottom'] ) ) {
			remove_theme_mod( 'footer_builder_footer_bottom' );
		} else {
			set_theme_mod( 'footer_builder_footer_bottom', 1 );
		}

		// Footer Widgets - Disabled by default
		if ( empty( $options['footer_builder_footer_widgets'] ) ) {
			remove_theme_mod( 'footer_builder_footer_widgets' );
		} else {
			set_theme_mod( 'footer_builder_footer_widgets', 1 );
		}

		// Update fixed footer - Disabled by default
		if ( empty( $options['fixed_footer'] ) ) {
			remove_theme_mod( 'fixed_footer' );
		} else {
			set_theme_mod( 'fixed_footer', 1 );
		}

		// Update footer Reveal - Disabled by default
		if ( empty( $options['footer_reveal'] ) ) {
			remove_theme_mod( 'footer_reveal' );
		} else {
			set_theme_mod( 'footer_reveal', true );
		}

		// Update bg
		if ( empty( $options['bg'] ) ) {
			remove_theme_mod( 'footer_builder_bg' );
		} else {
			set_theme_mod( 'footer_builder_bg', wp_strip_all_tags( $options['bg'] ) );
		}

		// Update bg img
		if ( empty( $options['bg_img'] ) ) {
			remove_theme_mod( 'footer_builder_bg_img' );
		} else {
			set_theme_mod( 'footer_builder_bg_img', wp_strip_all_tags( $options['bg_img'] ) );
		}

		// Update bg img style
		if ( empty( $options['bg_img_style'] ) ) {
			remove_theme_mod( 'footer_builder_bg_img_style' );
		} else {
			set_theme_mod( 'footer_builder_bg_img_style', wp_strip_all_tags( $options['bg_img_style'] ) );
		}

	}

	/**
	 * Main Settings section callback.
	 */
	public function section_main_callback( $options ) {
		// Leave blank
	}

	/**
	 * Fields callback functions.
	 */

	// Footer Builder Page ID
	public function content_id_field_callback() {
		$selected_template = get_theme_mod( 'footer_builder_page_id' );
		$template_exists   = get_post_status( $selected_template );
		$selected_exists   = false;

		$builder_template_types = Theme_Builder::get_template_post_types();
		?>

		<select name="footer_builder[content_id]" id="wpex-footer-builder-select" class="wpex-chosen">

			<?php if ( $selected_template && ! $template_exists ) { ?>
				<option value="" selected="selected"><?php esc_html_e( 'None', 'total' ); ?></option>
			<?php } else { ?>
				<option value=""><?php esc_html_e( 'None', 'total' ); ?></option>
			<?php } ?>

			<?php foreach ( $builder_template_types as $type => $type_name ) {

				$get_templates = new WP_Query( [
					'posts_per_page' => -1,
					'post_type'      => $type,
					'fields'         => 'ids',
				] );

				if ( ! empty( $get_templates->posts ) ) { ?>
					<optgroup label="<?php echo esc_attr( $type_name ); ?>"><?php
						foreach ( $get_templates->posts as $template_id ) {
							if ( ! $selected_exists && ( $selected_template == $template_id ) ) {
								$selected_exists = true;
							}
							echo '<option value="' . absint( $template_id ) . '"' . selected( $selected_template, $template_id, false ) . '>' . esc_html( get_the_title( $template_id ) ) . '</option>';
						}
					?></optgroup>
				<?php }

			}

			if ( $selected_template && $template_exists && ! $selected_exists ) {
				echo '<optgroup label="' . esc_attr( 'Other', 'total' ) . '"><option value="' . absint( $selected_template ) . '" selected="selected">' . esc_html( get_the_title( $selected_template ) ) . '</optgroup></option>';
			}

		?></select>

		<br><br>

		<?php if ( post_type_exists( 'wpex_templates' ) ) { ?>
			<div class="wpex-create-new-template">
				<a href="<?php echo esc_url( admin_url( 'post-new.php?post_type=wpex_templates' ) ); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html__( 'Create new template', 'total' ); ?></a>
			</div>
		<?php } ?>

		<span class="wpex-edit-template-links-spinner"><?php wpex_svg( 'loaders/wordpress', 20 ); ?></span>

		<div class="wpex-edit-template-links-ajax" data-nonce="<?php echo wp_create_nonce( 'wpex_footer_builder_edit_links_nonce' ); ?>" data-action="wpex_footer_builder_edit_links"><?php $this->edit_links( $selected_template ); ?></div>

	<?php }

	/**
	 * Footer Bottom Callback.
	 */
	public function footer_builder_footer_bottom_field_callback() {

		$val = get_theme_mod( 'footer_builder_footer_bottom', false ) ? 'on' : false;

		?>

		<input type="checkbox" name="footer_builder[footer_builder_footer_bottom]" id="wpex-footer-builder-footer-bottom" <?php checked( $val, 'on' ); ?>>

		<?php
	}

	/**
	 * Fixed Footer Callback.
	 */
	public function fixed_footer_field_callback() {
		$val = get_theme_mod( 'fixed_footer', false ) ? 'on' : false;

		?>

		<input type="checkbox" name="footer_builder[fixed_footer]" id="wpex-footer-builder-fixed" <?php checked( $val, 'on' ); ?>>

		<?php
	}

	/**
	 * Footer Reveal Callback.
	 */
	public function footer_reveal_field_callback() {
		$val = get_theme_mod( 'footer_reveal' ) ? 'on' : false;

		?>

		<input type="checkbox" name="footer_builder[footer_reveal]" id="wpex-footer-builder-reveal" <?php checked( $val, 'on' ); ?>>

		<?php
	}

	/**
	 * Footer Widgets Callback.
	 */
	public function footer_widgets_field_callback() {
		$val = get_theme_mod( 'footer_builder_footer_widgets', false ) ? 'on' : false;

		?>

		<input type="checkbox" name="footer_builder[footer_builder_footer_widgets]" id="wpex-footer-builder-widgets" <?php checked( $val, 'on' ); ?>>

		<?php
	}

	// Background Setting
	public function bg_field_callback() {
		?>

		<input id="wpex-footer-builder-bg" type="text" name="footer_builder[bg]" value="<?php echo esc_attr( get_theme_mod( 'footer_builder_bg' ) ); ?>" class="wpex-color-field">

	<?php }

	// Background Image Setting
	public function bg_img_field_callback() {
		$bg = get_theme_mod( 'footer_builder_bg_img' );

		?>

		<div class="uploader">
			<input id="wpex-footer-builder-bg-img" class="wpex-media-input" type="text" name="footer_builder[bg_img]" value="<?php echo esc_attr( $bg ); ?>">
			<button class="wpex-media-upload-button button-primary"><?php esc_attr_e( 'Select', 'total' ); ?></button>
			<button class="wpex-media-remove button-secondary"><?php esc_html_e( 'Remove', 'total' ); ?></button>
			<div class="wpex-media-live-preview">
				<?php if ( $preview = wpex_get_image_url( $bg ) ) { ?>
					<img src="<?php echo esc_url( $preview ); ?>" alt="<?php esc_html_e( 'Preview Image', 'total' ); ?>">
				<?php } ?>
			</div>
		</div>
	<?php }

	// Background Image Style Setting
	public function bg_img_style_field_callback() {
		$style = get_theme_mod( 'footer_builder_bg_img_style' );

		?>

		<select id="wpex-footer-builder-bg-style" name="footer_builder[bg_img_style]">
			<?php
			$bg_styles = wpex_get_bg_img_styles();
			foreach ( $bg_styles as $key => $val ) { ?>
				<option value="<?php echo esc_attr( $key ); ?>" <?php selected( $style, $key, true ); ?>>
					<?php echo strip_tags( $val ); ?>
				</option>
			<?php } ?>
		</select>
	<?php }

	/**
	 * Settings page output.
	 */
	public function create_admin_page() {
		wp_enqueue_media();

		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'wp-color-picker' );

		wp_enqueue_style( 'wpex-chosen' );
		wp_enqueue_script( 'wpex-chosen' );

		wp_enqueue_style( 'wpex-admin-pages' );
		wp_enqueue_script( 'wpex-admin-pages' );

		?>

		<div id="wpex-admin-page" class="wrap">

			<h1><?php esc_html_e( 'Footer Builder', 'total' ); ?></h1>

			<p><?php echo esc_html__( 'By default the footer consists of a simple widgetized area which you can control via the WordPress Customizer. For more complex layouts you can use the option below to select a template and create your own custom footer layout from scratch.', 'total' ); ?></p>

			<hr>

			<?php
			// Warning if footer builder page doesn't exist
			$page_id = get_theme_mod( 'footer_builder_page_id' );
			if ( $page_id && ( FALSE === get_post_status( $page_id ) || 'trash' === get_post_status( $page_id ) ) ) {

				echo '<div class="notice notice-warning"><p>' . esc_html__( 'It appears the page you had selected has been deleted, please re-save your settings to prevent issues.', 'total' ) . '</p></div>';

			} ?>

			<form method="post" action="options.php">

				<?php settings_fields( 'wpex_footer_builder' ); ?>
				<?php do_settings_sections( 'wpex-footer-builder-admin' ); ?>
				<?php submit_button(); ?>
			</form>

		</div>
	<?php }

	/**
	 * Alters get template.
	 */
	public function builder_template( $template ) {
		if ( empty( $_GET[ 'wpex_footer_template' ] ) ) {
			return $template;
		}
		$redirect = false;
		$current_post = absint( wpex_get_current_post_id() );
		$template_type = get_post_type( $current_post );
		if ( absint( $_GET[ 'wpex_footer_template' ] ) === $current_post
			&& in_array( $template_type, [ 'templatera', 'wpex_templates' ] )
		) {
			$redirect = true;
		} elseif ( $current_post === self::get_template_id() ) {
			$redirect = true;
		}
		if ( $redirect ) {
			$new_template = locate_template( 'single-' . $template_type . '.php' );
			if ( $new_template ) {
				return $new_template;
			}
		}
		return $template;
	}

	/**
	 * Alter the footer.
	 */
	public function alter_footer() {
		$this->insert_hook     = apply_filters( 'wpex_footer_builder_insert_hook', 'wpex_hook_footer_before' );
		$this->insert_priority = apply_filters( 'wpex_footer_builder_insert_priority', 40 );

		add_action( $this->insert_hook, array( $this, 'get_part' ), $this->insert_priority );
	}

	/**
	 * Add footer builder to array of ID's with CSS to load site-wide.
	 */
	public function wpex_vc_css_ids( $ids ) {
		$footer_builder_id = self::get_template_id(); // Get translated footer ID
		if ( $footer_builder_id ) {
			$ids[] = $footer_builder_id;
		}
		return $ids;
	}

	/**
	 * Remove customizer panels.
	 */
	public function remove_customizer_panels( $panels ) {
		if ( ! get_theme_mod( 'footer_builder_footer_widgets', false ) ) {
			unset( $panels['footer_widgets'] );
		}
		if ( ! get_theme_mod( 'footer_builder_footer_bottom', false ) ) {
			unset( $panels['footer_bottom'] );
		}
		return $panels;
	}

	/**
	 * Remove customizer settings.
	 */
	public function alter_customizer_settings( $sections ) {
		if ( get_theme_mod( 'footer_builder_footer_widgets', false ) ) {
			unset( $sections['wpex_footer_widgets']['settings']['footer_widgets'] );
			unset( $sections['wpex_footer_widgets']['settings']['fixed_footer'] );
			unset( $sections['wpex_footer_widgets']['settings']['footer_reveal'] );
		}
		if ( get_theme_mod( 'footer_builder_footer_bottom', false ) ) {
			unset( $sections['wpex_footer_bottom']['settings']['footer_bottom'] );
		}
		return $sections;
	}

	/**
	 * Remove typography settings.
	 *
	 * @since 4.7.1
	 */
	public function remove_typography_settings( $settings ) {
		if ( ! get_theme_mod( 'footer_builder_footer_widgets', false ) ) {
			unset( $settings['footer_widgets'] );
			unset( $settings['footer_widget_title'] );
		}
		if ( ! get_theme_mod( 'footer_builder_footer_bottom', false ) ) {
			unset( $settings['copyright'] );
			unset( $settings['footer_menu'] );
		}
		return $settings;
	}

	/**
	 * Gets the footer builder template part if the footer is enabled.
	 */
	public function get_part() {
		if ( Footer::is_enabled() ) {
			get_template_part( 'partials/footer/footer-builder' );
		}
	}

	/**
	 * Custom CSS for footer builder.
	 */
	public function wpex_head_css( $css ) {
		$add_css = '';
		if ( $bg = get_theme_mod( 'footer_builder_bg' ) ) {
			$add_css .= 'background-color:' . wp_strip_all_tags( $bg ) . ';';
		}
		if ( $bg_img = wpex_get_image_url( get_theme_mod( 'footer_builder_bg_img' ) ) ) {
			$add_css .= 'background-image:url(' . esc_url( $bg_img ) . ');';
		}
		if ( $bg_img && $bg_img_style = wpex_sanitize_data( get_theme_mod( 'footer_builder_bg_img_style' ), 'background_style_css' ) ) {
			$add_css .= $bg_img_style;
		}
		if ( $add_css ) {
			$add_css = '#footer-builder{ ' . $add_css . '}';
			$css .= '/*FOOTER BUILDER*/' . $add_css;
		}
		return $css;
	}

	/**
	 * Get edit links.
	 */
	public function edit_links( $template_id = '' ) {
		if ( ! $template_id ) {
			return;
		}

		$template_type = get_post_type( $template_id );

		?>

		<a href="<?php echo esc_url( admin_url( 'post.php?post=' . intval( $template_id ) . '&action=edit' ) ); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html__( 'Backend Edit', 'total' ); ?></a>

		<?php if ( WPEX_VC_ACTIVE && in_array( $template_type, [ 'templatera', 'wpex_templates' ] ) ) { ?>

		&vert; <a href="<?php echo esc_url( admin_url( 'post.php?vc_action=vc_inline&post_id=' . $template_id . '&post_type=' . get_post_type( $template_id ) . '&wpex_inline_footer_template_editor=' . $template_id ) ); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Frontend Edit', 'total' ); ?></a>

		<?php } ?>
	<?php }

	/**
	 * Return correct edit links.
	 */
	public function ajax_edit_links() {
		if ( empty( $_POST['template_id'] ) || ! wp_verify_nonce( $_POST['nonce'], 'wpex_footer_builder_edit_links_nonce' ) ) {
			wp_die();
		}

		$this->edit_links( absint( $_POST['template_id'] ) );

		wp_die();
	}

}
new FooterBuilder();