<?php
namespace TotalTheme;

defined( 'ABSPATH' ) || exit;

/**
 * Meta Class.
 *
 * @package TotalTheme
 * @subpackage Page
 * @version 5.6.1
 */
class Meta {

	/**
	 * Returns the array of registered meta blocks.
	 */
	public static function registered_blocks() {
		$blocks = [
			'date'           => esc_html__( 'Published Date', 'total' ),
			'date-modified'  => esc_html__( 'Modified Date', 'total' ),
			'author'         => esc_html__( 'Author', 'total' ),
			'categories'     => esc_html__( 'Categories', 'total' ),
			'first_category' => esc_html__( 'First Category', 'total' ),
			'comments'       => esc_html__( 'Comments', 'total' ),
		];

		return $blocks;
	}

	/**
	 * Render meta blocks.
	 */
	protected static function render_blocks( $args = [] ) {

		/**
		 * Filters the Meta args.
		 *
		 * @param array $args
		 */
		$args = (array) apply_filters( 'wpex_meta_args', $args );

		$blocks = $args['blocks'] ?? [];

		if ( ! $blocks ) {
			return;
		}

		foreach ( $blocks as $key => $val ) {

			if ( is_string( $val ) && array_key_exists( $val, self::registered_blocks() ) ) {
				$block_type = $val; // fixes issues where blocks array doesn't have defined keys.
			} else {
				$block_type = $key;
			}

			$block_args = [
				'icon'      => self::get_block_icon( $block_type ),
				'singular'  => $args['singular'] ?? true,    // used for schema markup.
				'hook_name' => $args['hook_name'] ?? 'meta',  // used for apply_filters.
			];

			switch ( $val ) {
				case 'meta':
					// prevent infinite loop with get_template_part()
					break;
				case 'date':
					$block_args['format'] = $args['date_format'] ?? '';
					get_template_part( 'partials/meta/blocks/date', null, $block_args );
					break;
				case 'date-modified':
					$block_args['format'] = $args['date_format'] ?? '';
					get_template_part( 'partials/meta/blocks/date-modified', null, $block_args );
					break;
				case 'author':
					$block_args['link'] = $args['author_link'] ?? true;
					get_template_part( 'partials/meta/blocks/author', null, $block_args );
					break;
				case 'categories':
					$taxonomy = $args['categories_tax'] ?? apply_filters( 'wpex_meta_categories_taxonomy', wpex_get_post_type_cat_tax() );
					if ( $taxonomy ) {
						$block_args['taxonomy'] = $taxonomy;
						get_template_part( 'partials/meta/blocks/categories', null, $block_args );
					}
					break;
				case 'first_category':
					$taxonomy = $args['first_category_tax'] ?? $args['categories_tax'] ?? apply_filters( 'wpex_meta_first_category_taxonomy', wpex_get_post_type_cat_tax() );
					if ( $taxonomy ) {
						$block_args['taxonomy'] = $taxonomy;
						get_template_part( 'partials/meta/blocks/first-category', null, $block_args );
					}
					break;
				case 'comments':
					get_template_part( 'partials/meta/blocks/comments', null, $block_args );
					break;
				default:
					$block_args['block_type']      = $key;
					$block_args['render_callback'] = $val;
					get_template_part( 'partials/meta/blocks/custom', null, $block_args );
					break;
			}
		} // end foreach
	}

	/**
	 * Return icon name for meta block.
	 */
	protected static function get_block_icon( $block_type = '' ) {
		$icon = '';

		switch( $block_type ) {
			case 'date':
			case 'date-modified':
				$icon = 'clock-o';
				break;
			case 'author':
				$icon = 'user-o';
				break;
			case 'categories':
			case 'first_category':
				$icon = 'folder-o';
				break;
			case 'comments':
				$icon = 'comment-o';
				break;
		}

		/**
		 * Filters the meta block icon.
		 *
		 * @param string $icon
		 * param string $block_type
		 */
		$icon = (string) apply_filters( 'wpex_meta_block_icon', $icon, $block_type );

		return $icon;
	}

}