<?php
namespace TotalTheme\Integration\WPBakery;

defined( 'ABSPATH' ) || exit;

class Section_Templates {

	/**
	 * Instance.
	 */
	private static $instance;

	/**
	 * Template Type Name.
	 */
	public const TEMPLATE_TYPE = 'wpex_section_templates';

	/**
	 * Create or retrieve the instance of Register_Section_Templates.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
			static::$instance->init_hooks();
		}
		return static::$instance;
	}

	/**
	 * Add hooks.
	 */
	protected function init_hooks() {

		// Register templates.
		add_action( 'vc_get_all_templates', [ $this, 'register_templates' ] );

		// Display templates in editor.
		add_action( 'vc_templates_render_category', [ $this, 'render_template_tab' ] );

		// Render templates.
		add_action( 'vc_templates_render_backend_template', [ $this, 'render_backend_template' ], 10, 2 );
		add_action( 'vc_templates_render_frontend_template', [ $this, 'render_frontend_template' ], 10, 2 );

		// Enqueue filter scripts.
		add_action( 'vc_frontend_editor_enqueue_js_css', [ $this, 'filter_js' ] );
		add_action( 'vc_backend_editor_enqueue_js_css', [ $this, 'filter_js' ] );
	}

	/**
	 * Returns current category name.
	 */
	public function get_category_name() {
		return esc_html__( 'Section Templates', 'total' );
	}

	/**
	 * Returns current category name.
	 */
	public function get_category_description() {
		return esc_html__( 'Append a section template to the current layout.', 'total' );
	}

	/**
	 * Register templates.
	 */
	public function register_templates( $data ) {
		$templates = $this->get_templates();

		if ( ! $templates ) {
			return $data;
		}

		$total_cat = [
			'category'             => self::TEMPLATE_TYPE,
			'category_name'        => $this->get_category_name(),
			'category_description' => $this->get_category_description(),
			'category_weight'      => apply_filters( 'wpex_vc_section_templates_category_weight', 11 ),
		];

		$total_cat_templates = [];

		$count = 0;
		foreach ( $templates as $template_id => $template_data ) {
			$count ++;
			$total_cat_templates[] = [
				'unique_id' => $template_id,
				'name'      => $template_data['name'],
				'type'      => self::TEMPLATE_TYPE,
				'content'   => $template_data['content'],
				'weight'    => $count,
			];
		}

		$total_cat['templates'] = $total_cat_templates;

		$data[] = $total_cat;

		return $data;
	}

	/**
	 * Get templates.
	 */
	protected function get_templates() {
		$templates = [];

		// Placeholder images.
		$ph_location  = TRAILINGSLASHIT( WPEX_THEME_URI ) . '/inc/integration/wpbakery/templates/placeholders/';
		$ph_landscape = $ph_location . 'landscape.png';
		$ph_portrait  = $ph_location . 'portrait.png';
		$ph_square    = $ph_location . 'square.png';

		// Loop through categories to get templates.
		$categories = $this->get_categories();

		if ( $categories ) {
			foreach ( $categories as $key => $val ) {
				$file = WPEX_INC_DIR . 'integration/wpbakery/templates/sections/' . $key . '.php';
				if ( file_exists( $file ) ) {
					require $file;
				}
			}
		}

		/**
		 * Filters the Total theme wpbakery section templates.
		 *
		 * @param array $templates.
		 */
		$templates = (array) apply_filters( 'wpex_wpbakery_section_templates', $templates );

		return $templates;
	}

	/**
	 * Returns template category.
	 */
	protected function get_template_category( $template_id = '' ) {
		$category = trim( preg_replace( '/[^a-z]/', '', $template_id ) );

		if ( 'calltoaction' === $category ) {
			$category = 'call-to-action';
		}

		return $category;
	}

	protected function slugify( $str = '' ) {
		if ( function_exists( 'vc_slugify' ) ) {
			return vc_slugify( $str );
		}

		$str = strtolower( $str );
		$str = html_entity_decode( $str );
		$str = preg_replace( '/[^\w ]+/', '', $str );
		$str = preg_replace( '/ +/', '-', $str );
		return $str;
	}

	/**
	 * Get templates JSON.
	 */
	protected function get_templates_json() {
		$templates = $this->get_templates();

		$templates_data = [];

		foreach ( $templates as $template_id => $template_data ) {
			$template = [
				'id'         => $template_id,
				'unique_id'  => $template_id,
				'id_hash'    => md5( $template_id ),
				'label'      => $template_data['name'],
				'name'       => $this->slugify( $template_data['name'] ),
				'category'   => $this->get_template_category( $template_id ),
				'screenshot' => TRAILINGSLASHIT( WPEX_THEME_URI ) . '/inc/integration/wpbakery/templates/sections/thumbnails/' . $template_id . '.png',
			];
			$template = array_map( 'esc_attr', $template );
			$templates_data[] = $template;
		}

		return wp_json_encode( $templates_data );
	}

	/**
	 * Render the template tab.
	 */
	public function render_template_tab( $category ) {
		if ( self::TEMPLATE_TYPE === $category['category'] ) {
			$category['output'] = $this->get_vc_tab_template();
		}
		return $category;
	}

	/**
	 * Get categories.
	 */
	protected function get_categories() {
		$categories = [
			'hero'           => esc_html__( 'Hero', 'total' ),
			'features'       => esc_html__( 'Features', 'total' ),
			'statistics'     => esc_html__( 'Statistics', 'total' ),
			'team'           => esc_html__( 'Team', 'total' ),
			'call-to-action' => esc_html__( 'Call to Action', 'total' ),
			'faq'            => esc_html__( 'FAQ', 'total' ),
			'subscribe'      => esc_html__( 'Subscribe', 'total' ),
			'pricing'        => esc_html__( 'Pricing', 'total' ),
			'contact'        => esc_html__( 'Contact', 'total' ),
		];

		/**
		 * Filters the Total theme wpbakery section template categories.
		 *
		 * @param array $templates.
		 */
		$categories = (array) apply_filters( 'wpex_wpbakery_section_templates_categories', $categories );

		return $categories;
	}

	/**
	 * Renders the items for the section templates tab.
	 *
	 * @param $template
	 * @return string
	 */
	protected function get_vc_tab_template() {
		ob_start();
			require_once WPEX_INC_DIR . 'integration/wpbakery/templates/category.tpl.php';
			return ob_get_clean();
	}

	/**
	 * Renders the items for the section templates tab.
	 *
	 * @param $template
	 * @return string
	 */
	protected function render_template_tab_item( $template ) {
		$name                = $template['name'];
		$template_id         = $template['unique_id'];
		$template_id_hash    = md5( $template_id );
		$template_name       = $name;
		$template_name_lower = function_exists( 'vc_slugify' ) ? vc_slugify( $template_name ) : '';
		$template_type       = self::TEMPLATE_TYPE;
		$template_category   = $this->get_template_category( $template_id );

		$preview_image = TRAILINGSLASHIT( WPEX_THEME_URI ) . '/inc/integration/wpbakery/templates/sections/thumbnails/' . $template_id . '.png';

			$output .= '<div class="wpex-vc-template-list__item"
						data-template_id="' . esc_attr( $template_id ) . '"
						data-template_id_hash="' . esc_attr( $template_id_hash ) . '"
						data-category="' . esc_attr( $template_type ) . '"
						data-template_unique_id="' . esc_attr( $template_id ) . '"
						data-template_name="' . esc_attr( $template_name_lower ) . '"
						data-template_type="' . esc_attr( $template_type ) . '"
						data-wpex-category="' . esc_attr( $template_category ) . '"
					>';

			if ( $preview_image ) {
				$output .= '<div class="wpex-vc-template-list__image"><img loading="lazy" src="' . esc_url( $preview_image ) . '"></div>';
			}

			$output .= '<div class="wpex-vc-template-list__overlay">';

			$output .= '<div class="wpex-vc-template-list__name">' . esc_html( $template_name ) . '</div>';

				$output .= '<div class="wpex-vc-template-list__actions">';

					$output .= '<a href="https://total.wpexplorer.com/sections/wpbakery/' . esc_attr( $template_id ) . '" class="button button-primary" target="_blank" rel="nofollow noopener noreferrer">' . esc_html__( 'Preview', 'total' ) . '<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 0 24 24" width="24px" fill="currentColor"><path d="M0 0h24v24H0V0z" fill="none"/><path d="M12 6.5c3.79 0 7.17 2.13 8.82 5.5-1.65 3.37-5.02 5.5-8.82 5.5S4.83 15.37 3.18 12C4.83 8.63 8.21 6.5 12 6.5m0-2C7 4.5 2.73 7.61 1 12c1.73 4.39 6 7.5 11 7.5s9.27-3.11 11-7.5c-1.73-4.39-6-7.5-11-7.5zm0 5c1.38 0 2.5 1.12 2.5 2.5s-1.12 2.5-2.5 2.5-2.5-1.12-2.5-2.5 1.12-2.5 2.5-2.5m0-2c-2.48 0-4.5 2.02-4.5 4.5s2.02 4.5 4.5 4.5 4.5-2.02 4.5-4.5-2.02-4.5-4.5-4.5z"/></svg></a>';

					$output .= '<button type="button" class="wpex-vc-template-list__insert button button-primary" data-template-handler="">' . esc_html__( 'Insert', 'total' ) . '<svg xmlns="http://www.w3.org/2000/svg" enable-background="new 0 0 24 24" height="24px" viewBox="0 0 24 24" width="24px" fill="currentColor"><g><rect fill="none" height="24" width="24"/></g><g><path d="M18,15v3H6v-3H4v3c0,1.1,0.9,2,2,2h12c1.1,0,2-0.9,2-2v-3H18z M17,11l-1.41-1.41L13,12.17V4h-2v8.17L8.41,9.59L7,11l5,5 L17,11z"/></g></svg></button></div>';

				$output .= '</div>';

			$output .= '</div>';

		return $output;

	//	return '<script type="text/html" id="vc_template-item">' . $output . '</script>';
	}

	/**
	 * Render template for the backend editor.
	 */
	public function render_backend_template( $template_id, $template_type ) {
		if ( self::TEMPLATE_TYPE === $template_type ) {
			$templates = $this->get_templates();
			if ( isset( $templates[$template_id] ) ) {
				return trim( $templates[$template_id]['content'] );
			}
		}
		return $template_id;
	}

	/**
	 * Render template for the frontend editor.
	 */
	public function render_frontend_template( $template_id, $template_type ) {
		if ( self::TEMPLATE_TYPE === $template_type ) {
			$templates = $this->get_templates();
			if ( isset( $templates[$template_id] ) ) {
				vc_frontend_editor()->setTemplateContent( trim( $templates[$template_id]['content'] ) );
				vc_frontend_editor()->enqueueRequired();
				vc_include_template( 'editors/frontend_template.tpl.php', [
					'editor' => vc_frontend_editor(),
				] );
				die(); // important wp_die() causes the page to break - can't use.
			}
			wp_send_json_error( [
				'code' => 'Wrong ID or no Template found #3',
			] );
		}
		return $template_id;
	}

	/**
	 * Enqueues template filter js.
	 *
	 */
	public function filter_js() {
		wp_enqueue_script(
			'wpex-vc-section-templates',
			wpex_asset_url( 'js/dynamic/wpbakery/wpex-vc-section-templates.min.js' ),
			[ 'jquery' ],
			WPEX_THEME_VERSION,
			true
		);
	}

}