<?php
namespace TotalTheme\Integration;

use TotalTheme\Fonts\Font_Loader;
use TotalTheme\Theme_Icons;

defined( 'ABSPATH' ) || exit;

/**
 * Elementor Configuration Class
 *
 * @package TotalTheme
 * @subpackage Integration
 * @version 5.6.1
 */
class Elementor {

	/**
	 * The theme font group name.
	 */
	const FONT_GROUP_ID = 'total';

	/**
	 * Instance.
	 */
	private static $instance;

	/**
	 * Create or retrieve the instance of Elementor.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
		}

		return static::$instance;
	}

	/**
	 * Hook into actions and filters.
	 */
	public function __construct() {
		add_action( 'wp_enqueue_scripts', [ __CLASS__, 'front_css' ] );
		add_action( 'elementor/theme/register_locations', [ __CLASS__, 'register_locations' ] );
		add_action( 'elementor/frontend/after_enqueue_scripts', [ __CLASS__, 'editor_scripts' ] );
	//	add_action( 'elementor/editor/after_enqueue_styles', [ __CLASS__, 'editor_styles' ] );
	//	add_action( 'elementor/preview/enqueue_styles', [ __CLASS__, 'preview_styles' ] );

		// Custom fonts suppport.
		add_action( 'elementor/fonts/groups',  [ __CLASS__, 'font_groups' ] );
		add_action( 'elementor/fonts/additional_fonts', [ __CLASS__, 'additional_fonts' ] );
		add_action( 'elementor/fonts/print_font_links/' . self::FONT_GROUP_ID, [ __CLASS__, 'print_fonts' ] );

		// Theme Icons support.
		if ( Theme_Icons::is_enabled() ) {
			add_action( 'elementor/icons_manager/additional_tabs', [ __CLASS__, 'icons_manager_additional_tabs' ] );
		}
	}

	/**
	 * Enqueues front-end CSS.
	 */
	public static function front_css() {
		wp_enqueue_style(
			'wpex-elementor',
			wpex_asset_url( 'css/wpex-elementor.css' ),
			[ 'elementor-frontend' ],
			WPEX_THEME_VERSION
		);
	}

	/**
	 * Registers Elementor locations.
	 */
	public static function register_locations( $elementor_theme_manager ) {

		/**
		 * Filters whether the theme should register all core elementor locations via
		 * $elementor_theme_manager->register_all_core_location().
		 *
		 * @param bool $check
		 */
		$register_core_locations = (bool) apply_filters( 'total_register_elementor_locations', true );

		if ( $register_core_locations ) {
			$elementor_theme_manager->register_all_core_location();
		}

		$elementor_theme_manager->register_location( 'togglebar', [
			'label'           => esc_html__( 'Togglebar', 'total' ),
			'multiple'        => true,
			'edit_in_content' => false,
		] );

		$elementor_theme_manager->register_location( 'topbar', [
			'label'           => esc_html__( 'Top Bar', 'total' ),
			'multiple'        => true,
			'edit_in_content' => false,
		] );

		$elementor_theme_manager->register_location( 'page_header', [
			'label'           => esc_html__( 'Page Header', 'total' ),
			'multiple'        => true,
			'edit_in_content' => false,
		] );

		$elementor_theme_manager->register_location( 'footer_callout', [
			'label'           => esc_html__( 'Footer Callout', 'total' ),
			'multiple'        => true,
			'edit_in_content' => false,
		] );

		$elementor_theme_manager->register_location( 'footer_bottom', [
			'label'           => esc_html__( 'Footer Bottom', 'total' ),
			'multiple'        => true,
			'edit_in_content' => false,
		] );

	}

	/**
	 * Add Theme Font Group.
	 */
	public static function font_groups( $groups ) {
		if ( wpex_get_registered_fonts() ) {
			$groups[ self::FONT_GROUP_ID ] = 'Total';
		}
		return $groups;
	}

	/**
	 * Add Theme Font Options.
	 */
	public static function additional_fonts( $additional_fonts ) {
		$registered_fonts = wpex_get_registered_fonts();
		if ( ! $registered_fonts ) {
			return $additional_fonts;
		}
		foreach ( $registered_fonts as $font_name => $font_args ) {
			$additional_fonts[$font_name] = self::FONT_GROUP_ID;
		}
		return $additional_fonts;
	}

	/**
	 * Enqueue fonts.
	 */
	public static function print_fonts( $font ) {
		if ( $font ) {
			wpex_enqueue_font( $font, 'registered' );
		}
	}

	/**
	 * Enqueue JS in the editor.
	 */
	public static function editor_scripts() {
		if ( ! defined( 'VCEX_ELEMENTOR_INTEGRATION' )
			|| ! class_exists( 'Elementor\\Plugin' )
			|| ! \Elementor\Plugin::$instance->preview->is_preview_mode()
		) {
			return;
		}

		wp_enqueue_script(
			'wpex-elementor-preview',
			wpex_asset_url( 'js/dynamic/elementor/wpex-elementor-preview.min.js' ),
			[],
			WPEX_THEME_VERSION,
			true
		);
	}

	/**
	 * Enqueue CSS in the Editor.
	 */
	public static function editor_styles() {}

	/**
	 * Enqueue CSS for the preview panel.
	 */
	public static function preview_styles() {}

	/**
	 * Adds Theme Icons tab to Icons Manager.
	 */
	public static function icons_manager_additional_tabs( $tabs ) {
		$tabs['ticon'] = [
			'name'            => 'ticon',
			'label'           => esc_html__( 'Theme Icons', 'total' ),
			'url'             => Theme_Icons::get_css_url(),
		//	'enqueue'         => Theme_Icons::get_css_url(), // this breaks elementor for some reason.
			'prefix'          => 'ticon-',
			'displayPrefix'   => 'ticon',
			'labelIcon'       => 'ticon ticon-total-theme',
			'ver'             => WPEX_THEME_VERSION,
			'fetchJson'       => Theme_Icons::get_json_url(),
			'native'          => true,
			'render_callback' => [__CLASS__, 'render_theme_icon'],
		];
		return $tabs;
	}

	/**
	 * Callback function for rendering theme icons inside elementor.
	 */
	public static function render_theme_icon( $icon = [], $attributes = [], $tag = 'i' ) {
		if ( empty( $icon['value'] ) || empty( $icon['library'] ) || 'ticon' !== $icon['library'] ) {
			return;
		}
		return Theme_Icons::get_icon( $icon['value'], $attributes );
	}

}