<?php
namespace TotalTheme\Header;

use TotalTheme\Header\Core as Header;
use TotalTheme\Header\Logo as Header_Logo;
use TotalTheme\Header\Overlay as Overlay_Header;

defined( 'ABSPATH' ) || exit;

/**
 * Sticky Header.
 *
 * @package TotalTheme
 * @subpackage Header
 * @version 5.7.0
 */
class Sticky {

	/**
	 * Check if enabled or not.
	 */
	protected static $is_enabled;

	/**
	 * The sticky style.
	 */
	protected static $style;

	/**
	 * The sticky logo src.
	 */
	protected static $logo_src;

	/**
	 * Returns sticky header style.
	 */
	public static function style() {
		if ( ! is_null( self::$style ) ) {
			return self::$style;
		}

		if ( 'builder' === Header::style() ) {
			return 'standard'; // Header builder only supports standard.
		}

		// Get default style from customizer.
		$style = get_theme_mod( 'fixed_header_style', 'standard' );

		// Move old disabled style to new on/off switch.
		if ( 'disabled' === $style ) {
			set_theme_mod( 'fixed_header', false );
			remove_theme_mod( 'fixed_header_style' );
		}

		// Fallback style.
		if ( ! $style || 'disabled' === $style ) {
			$style = 'standard';
		}

		/**
		 * Filters the header style.
		 *
		 * @param string $style
		 */
		$style = (string) apply_filters( 'totaltheme/header/sticky/style', $style );

		/*** deprecated ***/
		$style = (string) apply_filters( 'wpex_sticky_header_style', $style );

		self::$style = $style;

		return self::$style;
	}

	/**
	 * Checks if the header is enabled or not.
	 */
	public static function is_enabled() {
		if ( wpex_vc_is_inline() ) {
			return false;
		}

		if ( ! is_null( self::$is_enabled ) ) {
			return self::$is_enabled;
		}

		$check = false;

		// Get current post id.
		$post_id = wpex_get_current_post_id();

		// Check meta first it should override any filter!
		if ( $post_id ) {
			$meta_check = get_post_meta( $post_id, 'wpex_sticky_header', true );
			if ( 'disable' === $meta_check ) {
				return false;
			} else if ( 'enable' === $meta_check ) {
				return true;
			}
		}

		// Get header style.
		$header_style = Header::style();

		// Sticky header for builder.
		if ( 'builder' === $header_style ) {
			$check = get_theme_mod( 'header_builder_sticky', false );
		}

		// Standard sticky header.
		else {

			// @note: we still need to check the style incase someone used filters to set the header style to disabled.
			$check = ( 'disabled' === self::style() ) ? false : get_theme_mod( 'fixed_header', true );

			if ( $check
				&& ! in_array( Header::style(), wpex_get_header_styles_with_sticky_support() )
				&& ! get_theme_mod( 'fixed_header_mobile' )
			) {
				$check = false;
			}
		}

		/**
		 * Filters whether the sticky header is enabled.
		 *
		 * @param bool $check
		 */
		$check = (bool) apply_filters( 'totaltheme/header/sticky/is_enabled', $check );

		/*** deprecated ***/
		$check = (bool) apply_filters( 'wpex_has_fixed_header', $check );

		self::$is_enabled = $check;

		return self::$is_enabled;
	}

	/**
	 * Returns the sticky header start position.
	 */
	public static function get_start_position() {
		$position = get_theme_mod( 'fixed_header_start_position' );

		if ( is_singular() ) {
			$meta_position = get_post_meta( get_the_ID(), 'fixed_header_start_position', true );
			if ( $meta_position ) {
				$position = $meta_position;
			}
		}

		/**
		 * Filters if the sticky header start position.
		 *
		 * @param string $position
		 */
		$position = (string) apply_filters( 'totaltheme/header/sticky/start_position', $position );

		/*** deprecated ***/
		$position = (string) apply_filters( 'wpex_sticky_header_start_position', $position );

		return $position;
	}

	/**
	 * Checks if the shrink header is enabled.
	 */
	public static function is_shrink_enabled() {
		$check = false;

		if ( in_array( self::style(), [ 'shrink', 'shrink_animated' ] ) ) {
			$check = true;
		}

		/**
		 * Filters if the sticky header is set to "strink".
		 *
		 * @param bool $check
		 * @deprecated 6.7.0
		 */
		$check = (bool) apply_filters( 'wpex_has_shrink_sticky_header', $check );

		return $check;
	}

	/**
	 * Checks if the shrink header is enabled on mobile.
	 */
	public static function is_shrink_enabled_mobile() {
		$check = true;

		/**
		 * Filters if the sticky header is set to "strink" on mobile
		 *
		 * @param bool $check
		 */
		$check = (bool) apply_filters( 'wpex_has_shrink_sticky_header_mobile', $check );

		return $check;
	}

	/**
	 * Returns sicky header logo image url.
	 */
	public static function get_logo_image_url() {
		if ( 'builder' === Header::style() ) {
			return ''; // Not needed for the sticky header builder.
		}

		$logo_src = self::get_logo_image_src();

		if ( isset( $logo_src[0] ) ) {
			return wpex_get_image_url( $logo_src[0]  );
		}
	}

	/**
	 * Returns sicky header logo image src.
	 */
	public static function get_logo_image_src() {
		if ( ! is_null( self::$logo_src ) ) {
			return self::$logo_src;
		}

		$logo = get_theme_mod( 'fixed_header_logo' );

		// Set sticky logo to header logo for overlay header when custom overlay logo is set
		// This way you can have a white logo on overlay but the default on sticky.
		if ( empty( $logo )
			&& ! Overlay_Header::is_global() // make sure the page is not using a global overlay header.
			&& Overlay_Header::logo_img() // check for custom overlay header logo.
			&& Overlay_Header::is_enabled() // check if overlay header is enabled.
		) {
			$header_logo = Header_Logo::get_image_id( false );
			if ( $header_logo ) {
				$logo = $header_logo;
			}
		}

		/**
		 * Filters the sticky header logo image.
		 *
		 * @param mixed string|int $logo
		 */
		$logo = apply_filters( 'totaltheme/header/sticky/logo_image_id', $logo );

		/*** deprecated ***/
		$logo = apply_filters( 'wpex_fixed_header_logo', $logo );

		if ( is_numeric( $logo ) ) {
			self::$logo_src = wp_get_attachment_image_src( $logo, 'full', false );
		} elseif ( is_string( $logo ) ) {
			self::$logo_src = array( $logo, '', '', '' );
		}

		return self::$logo_src;
	}

	/**
	 * Returns sicky header logo image height.
	 */
	public static function get_logo_image_height() {
		$logo_src = self::get_logo_image_src();
		if ( ! empty( $logo_src[2] ) ) {
			return absint( $logo_src[2] );
		} else {
			return Header_Logo::get_image_height();
		}
	}

	/**
	 * Returns sicky header logo image width.
	 */
	public static function get_logo_image_width() {
		$logo_src = self::get_logo_image_src();
		if ( ! empty( $logo_src[1] ) ) {
			return absint( $logo_src[1] );
		} else {
			return Header_Logo::get_image_width();
		}
	}

	/**
	 * Returns sicky header logo image url.
	 */
	public static function get_retina_logo_image_url() {
		$logo = wpex_get_translated_theme_mod( 'fixed_header_logo_retina' );

		/*
		 * Set retina logo for sticky header when the header overlay is set
		 * and the sticky header logo isn't set, since the default logo is displayed for the sticky header
		 * when using the overlay header and a custom logo.
		 */
		if ( ! $logo && ! get_theme_mod( 'fixed_header_logo' ) ) {

			$logo = wpex_get_translated_theme_mod( 'retina_logo' );

			/*** deprecated ***/
			$logo = apply_filters( 'wpex_header_logo_img_retina_url', $logo );
		}

		/**
		 * Filters the sticky header logo retina image.
		 *
		 * @param mixed $logo
		 */
		$logo = apply_filters( 'totaltheme/header/sticky/logo_retina_image_id', $logo );

		/*** deprecated ***/
		$logo = apply_filters( 'wpex_fixed_header_logo_retina', $logo );

		return wpex_get_image_url( $logo );
	}

}