<?php
namespace TotalTheme\Header;

use TotalTheme\Header\Overlay as Overlay_Header;
use TotalTheme\Header\Sticky as Sticky_Header;

defined( 'ABSPATH' ) || exit;

/**
 * Core header methods.
 *
 * @package TotalTheme
 * @subpackage Header
 * @version 5.6.1
 */
class Core {

	/**
	 * Header is enabled or not.
	 */
	protected static $is_enabled;

	/**
	 * Header is custom, aka using the header builder.
	 */
	protected static $is_custom;

	/**
	 * The header style.
	 */
	protected static $style;

	/**
	 * Checks if the header is enabled or not.
	 */
	public static function is_enabled() {
		if ( ! is_null( self::$is_enabled ) ) {
			return self::$is_enabled;
		}

		if ( is_page_template( 'templates/landing-page.php' ) ) {
			$check = false;
		} else {

			$post_id = wpex_get_current_post_id();

			if ( self::is_custom() || wpex_elementor_location_exists( 'header' ) ) {
				$check = true;
			} else {
				$check = get_theme_mod( 'enable_header', true );
			}

			if ( $post_id && $meta = get_post_meta( $post_id, 'wpex_disable_header', true ) ) {
				switch ( $meta ) {
					case 'on':
						$check = false;
						break;
					case 'enable':
						$check = true;
						break;
				}
			}
		}

		/**
		 * Filters whether the site has the header enabled.
		 *
		 * @param bool $check
		 * @todo rename to wpex_has_header for consistency.
		 */
		$check = (bool) apply_filters( 'totaltheme/header/is_enabled', $check );

		/*** deprecated ***/
		$check = (bool) apply_filters( 'wpex_display_header', $check );

		self::$is_enabled = $check;

		return self::$is_enabled;
	}

	/**
	 * Checks if the header is custom or not.
	 */
	public static function is_custom() {
		if ( ! is_null( self::$is_custom ) ) {
			return self::$is_custom;
		}

		self::$is_custom = false;

		if ( ! empty( wpex_header_builder_id() ) ) {
			self::$is_custom = true;
		}

		return self::$is_custom;
	}

	/**
	 * Returns the header style.
	 */
	public static function style() {
		if ( ! is_null( self::$style ) ) {
			return self::$style;
		}

		if ( ! self::is_enabled() ) {
			self::$style = 'disabled';
			return self::$style;
		}

		// Check if builder is enabled.
		if ( self::is_custom() ) {
			self::$style = 'builder';
			return self::$style;
		}

		// Get header style from customizer setting.
		$style = get_theme_mod( 'header_style', 'one' );

		/**
		 * Filters the list of header styles excluded for the overlay header.
		 *
		 * @param array $excluded_headers
		 */
		$excluded_overlay_header_styles = (array) apply_filters( 'wpex_overlay_header_excluded_header_styles', [] );

		if ( in_array( $style, $excluded_overlay_header_styles ) && Overlay_Header::is_enabled() ) {
			$style = 'one';
		}

		$post_id = wpex_get_current_post_id();

		// Check for custom header style defined in meta options => Overrides all.
		if ( 'dev' !== $style
			&& $post_id
			&& $meta = get_post_meta( $post_id, 'wpex_header_style', true ) ) {
			$style = $meta;
		}

		if ( ! $style ) {
			$style = 'one';
		}

		/**
		 * Filters the site header style.
		 *
		 * @param string $style
		 */
		$style = (string) apply_filters( 'totaltheme/header/style', $style );

		/*** deprecated ***/
		$style = (string) apply_filters( 'wpex_header_style', $style );

		self::$style = $style;

		return self::$style;
	}

	/**
	 * Checks if the header has a fixed height.
	 */
	public static function has_fixed_height() {
		$check        = false;
		$header_style = self::style();

		if ( in_array( $header_style, array( 'seven', 'eight', 'nine', 'ten' ) ) ) {
			$check = true;
		}

		/**
		 * Filters whether the site header header has a fixed height.
		 *
		 * @param bool $check
		 * @param string $header_style
		 */
		$check = (bool) apply_filters( 'totaltheme/header/has_fixed_height', $check, $header_style );

		/*** deprecated ***/
		$check = (bool) apply_filters( 'wpex_header_has_fixed_height', $check, $header_style );

		return $check;
	}

	/**
	 * Checks if the header has a flex container or not.
	 */
	public static function has_flex_container() {
		$check        = false;
		$header_style = self::style();

		if ( in_array( $header_style, array( 'seven', 'eight', 'nine', 'ten' ) ) ) {
			$check = true;
		}

		/**
		 * Filters whether the site header has a flex header container.
		 *
		 * @param bool $check
		 * @param string $header_style
		 */
		$check = (bool) apply_filters( 'totaltheme/header/has_flex_container', $check, $header_style );

		/*** deprecated ***/
		$check = (bool) apply_filters( 'wpex_has_flex_header', $check, $header_style );

		return $check;
	}

	/**
	 * Returns header image background image url.
	 */
	public static function get_background_image_url() {
		$image = get_theme_mod( 'header_background_image' );

		/**
		 * Filters the header background image.
		 *
		 * @param int|string $image
		 */
		$image = apply_filters( 'totaltheme/header/background_image', $image );

		/*** deprecated ***/
		$image = apply_filters( 'wpex_header_background_image', $image );

		$post_id = wpex_get_current_post_id();

		if ( $post_id && $meta_image = get_post_meta( $post_id, 'wpex_header_background_image', true ) ) {
			$image = $meta_image; // meta overrides filters.
		}

		return wpex_get_image_url( $image );
	}

	/**
	 * Return header wrapper classes.
	 *
	 * Provides a fallback for the older wpex_header_classes() function.
	 *
	 * @todo deprecate
	 */
	public static function get_wrapper_classes() {
		$post_id         = wpex_get_current_post_id();
		$header_style    = self::style();
		$has_flex_header = self::has_flex_container();

		// Setup classes array.
		$classes = [];

		// Main header style.
		$classes[] = 'header-' . sanitize_html_class( $header_style );

		// Builder classes
		if ( 'builder' === $header_style ) {

			// z-index prevents issues with vc rows and dropdowns from 3rd party menu plugins.
			// Also a z-index of at least 3 is needed when footer-reveal is enabled.
			$classes[] = 'wpex-z-10';
		}

		// Non-Builder classes.
		else {

			// Full width header.
			if ( 'full-width' === wpex_site_layout() && get_theme_mod( 'full_width_header' ) ) {
				$classes[] = 'header-full-width';
			}

			// Non-dev classes
			if ( 'dev' !== $header_style ) {

				// Fixed height class.
				if ( wpex_header_has_fixed_height( $header_style ) ) {
					$classes[] = 'header-fixed-height';
				}

				// Flex header style two.
				if ( 'two' === $header_style && get_theme_mod( 'header_flex_items', false ) ) {
					$classes[] = 'wpex-header-two-flex-v';
				}

				// Dropdown style (must be added here so we can target shop/search dropdowns).
				$dropdown_style = wpex_get_header_menu_dropdown_style();
				if ( $dropdown_style && 'default' !== $dropdown_style ) {
					$classes[] = 'wpex-dropdown-style-' . sanitize_html_class( $dropdown_style );
				}

				// Dropdown shadows.
				if ( $shadow = get_theme_mod( 'menu_dropdown_dropshadow' ) ) {
					$classes[] = 'wpex-dropdowns-shadow-' . sanitize_html_class( $shadow );
				}

			}

		}

		// Sticky Header.
		if ( Sticky_Header::is_enabled() ) {

			// Main fixed class.
			$classes[] = 'fixed-scroll';
			if ( ! in_array( $header_style, wpex_get_header_styles_with_sticky_support() ) ) {
				$classes[] = 'fixed-scroll--mobile-only';
			}

			$classes[] = 'wpex-z-99';
			if ( get_theme_mod( 'has_fixed_header_dropshadow', true ) ) {
				$classes[] = 'has-sticky-dropshadow';
			}
			if ( Sticky_Header::is_shrink_enabled() ) {
				$classes[] = 'shrink-sticky-header';
				if ( 'shrink_animated' === Sticky_Header::style() ) {
					$classes[] = 'anim-shrink-header';
				}
				if ( ! $has_flex_header ) {
					$classes[] = 'on-shrink-adjust-height';
				}
			}

		}

		// Header Overlay Style
		if ( Overlay_Header::is_enabled() ) {

			// Add overlay header class.
			$classes[] = 'overlay-header';

			// Add responsive class.
			if ( Overlay_Header::get_breakpoint() ) {
				$classes[] = 'overlay-header--responsive';
			}

			// Add overlay header style class.
			$overlay_style = Overlay_Header::style();

			if ( $overlay_style ) {
				$classes[] = sanitize_html_class( $overlay_style ) . '-style';
			}

		}

		// Custom bg.
		if ( get_theme_mod( 'header_background' ) ) {
			$classes[] = 'custom-bg';
		}

		// Background style.
		if ( self::get_background_image_url() ) {
			$bg_style = get_theme_mod( 'header_background_image_style' ) ?: '';

			/**
			 * Filters the header background image style.
			 *
			 * @param string $style
			 */
			$bg_style = apply_filters( 'totaltheme/header/background_image_style', $bg_style );

			/*** deprecated ***/
			$bg_style = apply_filters( 'wpex_header_background_image_style', $bg_style );

			if ( $bg_style ) {
				$classes[] = 'bg-' . sanitize_html_class( $bg_style );
			}

		}

		// Dynamic style class.
		$classes[] = 'dyn-styles';

		// Add relative class always.
		$classes[] = 'wpex-relative';  // !! important !!!

		// Clearfix class.
		if ( ! $has_flex_header ) {
			$classes[] = 'wpex-clr';
		}

		// Set keys equal to vals.
		$classes = array_combine( $classes, $classes );

		/**
		 * Filters the site header classes.
		 *
		 * @param array $classes
		 */
		$classes = (array) apply_filters( 'totaltheme/header/wrapper_class', $classes );

		/*** deprecated ***/
		$classes = (array) apply_filters( 'wpex_header_classes', $classes );

		// Turn classes into space seperated string.
		$classes = implode( ' ', $classes );

		return $classes;
	}

	/**
	 * Output header wrapper class.
	 */
	public static function wrapper_class() {
		$classes = self::get_wrapper_classes();
		if ( $classes ) {
			echo 'class="' . esc_attr( $classes ) . '"';
		}
	}

	/**
	 * Output header inner class.
	 */
	public static function inner_class() {
		$header_style     = self::style();
		$has_flex_header  = self::has_flex_container();
		$has_fixed_height = in_array( $header_style, array( 'seven', 'eight', 'nine', 'ten' ) );
		$add_clearfix     = true;

		$class = array(
			'header-' . sanitize_html_class( $header_style ) . '-inner',
		);

		if ( ! $has_fixed_height ) {
			$class[] = 'header-padding';
		}

		$class[] = 'container';

		/* Utility Classes */
		$class[] = 'wpex-relative';
		$class[] = 'wpex-h-100';

		if ( 'builder' !== $header_style ) {
			if ( $has_flex_header ) {
				$add_clearfix = false;
				$class[] = 'wpex-flex';
				$class[] = 'wpex-z-10'; // fixes issues with relative positioning and overflows, changed from z-2 to z-10 in 5.4.3
			} else {
				$class[] = 'wpex-py-30';
			}
		}

		if ( 'two' === $header_style && get_theme_mod( 'header_flex_items', false ) ) {
			$is_flex = true;
			$class[] = 'wpex-flex';
			$class[] = 'wpex-items-center';
		}

		if ( $add_clearfix ) {
			$class[] = 'wpex-clr';
		}

		/**
		 * Filters the footer bottom element class.
		 *
		 * @param array $class
		 */
		$class = (array) apply_filters( 'totaltheme/header/inner_class', $class );

		/*** deprecated ***/
		$class = (array) apply_filters( 'wpex_header_inner_class', $class );

		if ( $class ) {
			echo 'class="' . esc_attr( implode( ' ', $class ) ) . '"';
		}
	}

}