<?php
defined( 'ABSPATH' ) || exit;

/**
 * Sanitize data via the TotalTheme\SanitizeData class.
 */
function wpex_sanitize_data( $data = '', $type = '' ) {
	if ( $data && $type ) {
		$class = new TotalTheme\Sanitize_Data();
		return $class->parse_data( $data, $type );
	}
}

/**
 * Validate Boolean.
 */
function wpex_validate_boolean( $var ) {
	if ( is_bool( $var ) ) {
		return $var;
	}
	if ( is_string( $var ) ) {
		$var = strtolower( $var );
		if ( 'false' === $var || 'off' === $var || 'disabled' === $var || 'no' === $var ) {
			return false;
		}
		if ( 'true' === $var || 'on' === $var || 'enabled' === $var || 'yes' === $var ) {
			return true;
		}
	}
	return (bool) $var;
}

/**
 * Echo escaped post title.
 */
function wpex_esc_title( $post = '' ) {
	echo wpex_get_esc_title( $post );
}

/**
 * Return escaped post title.
 */
function wpex_get_esc_title( $post = '' ) {
	return the_title_attribute( [
		'echo' => false,
		'post' => $post,
	] );
}

/**
 * Wrapper for esc_attr with a fallback.
 */
function wpex_esc_attr( $val = null, $fallback = null ) {
	if ( $val = esc_attr( $val ) ) {
		return $val;
	}
	return $fallback;
}

/**
 * Wrapper for esc_html with a fallback.
 */
function wpex_esc_html( $val = null, $fallback = null ) {
	if ( $val = esc_html( $val ) ) {
		return $val;
	}
	return $fallback;
}

/**
 * Wrapper function for intval with a fallback.
 */
function wpex_intval( $val = null, $fallback = null ) {
	if ( 0 == $val ) {
		return 0; // Some settings may need this
	} elseif ( $val = intval( $val ) ) {
		return $val;
	} else {
		return $fallback;
	}
}

/**
 * Sanitize checkbox.
 */
function wpex_sanitize_checkbox( $checked ) {
	return ! empty( $checked ) ? true : false;
}

/**
 * Sanitize font-family.
 */
function wpex_sanitize_font_family( $font_family ) {
	if ( 'system-ui' === $font_family ) {
		$font_family = wpex_get_system_ui_font_stack();
	}
	$font_family = str_replace( "``", "'", $font_family ); // Fixes issue with fonts saved in shortcodes
	$font_family = wpex_get_font_family_stack( $font_family );
	return wp_specialchars_decode( $font_family );
}

/**
 * Sanitize visibility.
 */
function wpex_sanitize_visibility( $input ) {
	if ( empty( $input ) ) {
		return '';
	}
	$input = str_replace( '-portrait', '', $input );
	$input = str_replace( '-landscape', '', $input );
	return sanitize_html_class( $input );
}

/**
 * Sanitize visibility for the customizer
 */
function wpex_customizer_sanitize_visibility( $input, $setting = '' ) {
	if ( $input ) {
		$input = str_replace( '-portrait', '', $input );
		$input = str_replace( '-landscape', '', $input );
	}
	if ( 'always-visible' === $input ) {
		return '';
	}
	if ( array_key_exists( $input, wpex_visibility() ) ) {
		return $input;
	}
	if ( ! empty( $setting->default ) ) {
		return $setting->default;
	}
}

/**
 * Sanitize font-size customizer value which supports responsive sizing.
 */
function wpex_sanitize_font_size_mod( $input ) {
	// Sanitize array values
	if ( is_array( $input ) ) {
		return array_map( 'wpex_sanitize_font_size', $input );
	}
	// Convert jSON to array and sanitize each value while doing so
	if ( strpos( $input, '{' ) !== false ) {
		$input = json_decode( $input, true );
		return array_map( 'wpex_sanitize_font_size', $input );
	}
	return wpex_sanitize_font_size( $input ); // Most likely a string.
}

/**
 * Sanitize font-size.
 */
function wpex_sanitize_font_size( $input ) {
	if ( strpos( $input, 'px' )
		|| strpos( $input, 'em' )
		|| strpos( $input, 'vw' )
		|| strpos( $input, 'vmin' )
		|| strpos( $input, 'vmax' )
		// @todo allow for calc and clamp
	) {
		$input = esc_html( $input );
	} else {
		$input = absint( $input ) . 'px';
	}

	if ( $input !== '0px' && $input !== '0em' ) {
		return esc_html( $input );
	}
	return '';
}

/**
 * Sanitize letter spacing.
 */
function wpex_sanitize_letter_spacing( $input ) {
	if ( '' === $input ) {
		return '';
	}
	if ( is_numeric( $input ) ) {
		$input = floatval( $input ) . 'px'; // fallback to px.
	}
	return wp_strip_all_tags( $input );
}

/**
 * Sanitize line-height.
 */
function wpex_sanitize_line_height( $input ) {
	return wp_strip_all_tags( $input );
}

/**
 * Sanitize image.
 */
function wpex_sanitize_image( $input ) {
	return wp_kses( $input, [
		'img' => [
			'src'    => [],
			'alt'    => [],
			'srcset' => [],
			'id'     => [],
			'class'  => [],
			'height' => [],
			'width'  => [],
			'data'   => [],
		],
	] );
}

/**
 * Sanitize customizer select.
 */
function wpex_sanitize_customizer_select( $input, $setting ) {
	$input   = sanitize_key( $input );
	$choices = $setting->manager->get_control( $setting->id )->choices;
	if ( array_key_exists( $input, $choices ) ) {
		return sanitize_text_field( $input );
	}
	if ( ! empty( $setting->default ) ) {
		return $setting->default;
	}
}

/**
 * Sanitize customizer columns control.
 */
function wpex_sanitize_customizer_columns( $input, $setting ) {
	if ( is_numeric( $input ) ) {
		return absint( $input );
	}
	$input = (array) $input;
	return array_map( 'absint', $input );
}

/**
 * Sanitize Template Content.
 */
function wpex_sanitize_template_content( $template_content = '' ) {
	if ( $template_content ) {
		return wpex_the_content( $template_content );
	}
}

/**
 * Parse visual composer content to remove extra p tags.
 *
 * @todo deprecated 5.7.0 - use wpex_clean_up_shortcodes instead.
 */
function wpex_parse_vc_content( $content = '' ) {
	if ( $content ) {
		// Fix sections with p tags around them.
		$content = str_replace( '<p>[vc_section', '[vc_section', $content );
		$content = str_replace( '[/vc_section]</p>', '[/vc_section]', $content );

		// Fix rows with p tags around them.
		$content = str_replace( '<p>[vc_row', '[vc_row', $content );
		$content = str_replace( '[/vc_row]</p>', '[/vc_row]', $content );
	}
	return $content;
}

/**
 * Clean up shortcodes.
 */
function wpex_clean_up_shortcodes( $content = '' ) {
	if ( $content ) {
		return strtr( $content, [
			'<p>['    => '[',
			']</p>'   => ']',
			']<br />' => ']'
		] );
	}
}