<?php
use TotalTheme\Header\Core as Header;
use TotalTheme\Header\Menu as Header_Menu;
use TotalTheme\Mobile\Menu as Mobile_Menu;
use TotalTheme\Header\Overlay as Overlay_Header;

defined( 'ABSPATH' ) || exit;

/*-------------------------------------------------------------------------------*/
/* [ Table of contents ]
/*-------------------------------------------------------------------------------*
	# General
	# Dropdowns
	# Drop Widgets
	# Mobile
/*-------------------------------------------------------------------------------*/
/* [ General ]
/*-------------------------------------------------------------------------------*/

/**
 * Check if header has a menu.
 */
function wpex_has_header_menu() {
	if ( ! wpex_has_header() || wpex_has_custom_header() ) {
		return false;
	}

	// Check meta first.
	$meta_check = get_post_meta( wpex_get_current_post_id(), 'wpex_header_menu', true );
	if ( $meta_check ) {
		return wpex_validate_boolean( $meta_check );
	}

	$check = false;

	// Menu Location.
	$menu_location = wpex_header_menu_location();

	// Custom menu.
	$custom_menu = wpex_custom_menu();

	// Multisite global menu.
	$ms_global_menu = (bool) apply_filters( 'wpex_ms_global_menu', false );

	// We have a menu defined so set check to true.
	if ( has_nav_menu( $menu_location ) || $custom_menu || $ms_global_menu ) {
		$check = true;
	}

	/**
	 * Filters whether the header has menu.
	 *
	 * @param bool $check
	 */
	$check = (bool) apply_filters( 'wpex_has_header_menu', $check );

	return $check;
}

/**
 * Returns header menu location.
 */
function wpex_header_menu_location() {
	$location = 'main_menu';

	/**
	 * Filters the header menu location.
	 *
	 * @param string $location
	 */
	$location = (string) apply_filters( 'wpex_main_menu_location', $location );

	return $location;
}

/**
 * Return true if the header menu is sticky.
 */
function wpex_has_sticky_header_menu() {
	$check = false;
	$header_style = Header::style();

	if ( 'two' === $header_style || 'three' === $header_style || 'four' === $header_style ) {
		$check = get_theme_mod( 'fixed_header_menu', true );
	}

	/**
	 * Filters whether the header sticky header menu is enabled.
	 *
	 * @param bool $check
	 */
	$check = (bool) apply_filters( 'wpex_has_sticky_header_menu', $check );

	return $check;
}

/**
 * Returns menu classes.
 */
function wpex_header_menu_classes( $return = '' ) {
	$classes         = [];
	$header_style    = Header::style();
	$has_flex_header = Header::has_flex_container();
	$has_overlay     = Overlay_Header::is_enabled();
	$dropdown_style  = wpex_get_header_menu_dropdown_style();

	// Return wrapper classes.
	if ( 'wrapper' === $return ) {

		// Add Header Style to wrapper.
		$classes[] = 'navbar-style-' . sanitize_html_class( $header_style );

		// Fixed Height class.
		if ( 'one' === $header_style ) {
			$classes[] = 'navbar-fixed-height';
		}

		// 100% height class.
		if ( 'five' === $header_style ) {
			$classes[] = 'wpex-h-100';
		}

		// Line Height class.
		if ( in_array( $header_style, array( 'one', 'two', 'three', 'four', 'five' ) ) ) {
			if ( 'one' === $header_style ) {
				if ( ! get_theme_mod( 'menu_flush_dropdowns', false ) ) {
					$classes[] = 'navbar-fixed-line-height';
				}
			} else {
				$classes[] = 'navbar-fixed-line-height';
			}
		}

		// Add classes for the sticky header menu.
		if ( wpex_has_sticky_header_menu() ) {
			$classes[] = 'fixed-nav';
		}

		// Style specific classes.
		if ( 'dev' !== $header_style && ! wpex_is_header_menu_custom() ) {

			// Flex Header styles.
			if ( $has_flex_header ) {
				$classes[] = 'wpex-max-h-100';
				if ( 'seven' === $header_style || 'eight' === $header_style ) {
					$classes[] = 'wpex-mr-auto';
				} elseif ( 'nine' === $header_style ) {
					$classes[] = 'wpex-ml-auto';
				}
			}

			// Active underline.
			if ( 'six' !== $header_style && get_theme_mod( 'menu_active_underline', false ) ) {
				$classes[] = 'has-menu-underline';
			}

			// Dropdown caret.
			if ( ! get_theme_mod( 'menu_flush_dropdowns', false )
				&& ! get_theme_mod( 'menu_dropdown_top_border' )
				&& ! in_array( $dropdown_style, array( 'minimal-sq', 'minimal' ) )
				&& in_array( $header_style, array( 'one', 'five', 'seven', 'eight', 'nine', 'ten' ) )
			) {
				$has_caret = get_theme_mod( 'menu_dropdown_caret', true );
			} else {
				$has_caret = false;
			}

			/**
			 * Filters whether the header menu dropdowns have the caret enabled.
			 *
			 * @param bool $has_caret
			 */
			$has_caret = apply_filters( 'wpex_has_header_menu_dropdown_caret', $has_caret );

			if ( $has_caret ) {
				$classes[] = 'wpex-dropdowns-caret';
			}

			// Flush Dropdowns.
			if ( Header_Menu::has_flush_dropdowns() ) {
				$classes[] = 'wpex-flush-dropdowns';
				if ( $has_flex_header ) {
					$classes[] = 'wpex-self-stretch';
				}
			}

			// Add special class if the dropdown top border option in the admin is enabled.
			if ( get_theme_mod( 'menu_dropdown_top_border' ) ) {
				$classes[] = 'wpex-dropdown-top-border';
			}

			// Disable outer borders.
			if ( in_array( $header_style, array( 'two', 'three', 'four' ) )
				&& get_theme_mod( 'header_menu_disable_outline', false )
			) {
				$classes[] = 'no-outline';
			}

			// Disable inner borders.
			if ( in_array( $header_style, array( 'two', 'six' ) )
				&& get_theme_mod( 'header_menu_disable_borders', false )
			) {
				$classes[] = 'no-borders';
			}

			// Center items.
			if ( 'two' === $header_style && get_theme_mod( 'header_menu_center', false ) ) {
				$classes[] = 'center-items';
			}

			// Stretch items.
			if ( get_theme_mod( 'header_menu_stretch_items', false )
				&& in_array( $header_style, array( 'two', 'three', 'four', 'five' ) )
			) {
				$classes[] = 'wpex-stretch-items';
			}

		}

		// Stretch megamenus.
		if ( get_theme_mod( 'megamenu_stretch', true )
			&& ( 'one' === $header_style || $has_flex_header )
		) {
			$classes[] = 'wpex-stretch-megamenus';
		}

		// Add breakpoint class.
		if ( Mobile_Menu::is_enabled() ) {
			$classes[] = 'hide-at-mm-breakpoint';
		}

		// Add clearfix.
		if ( ! $has_flex_header ) {
			$classes[] = 'wpex-clr';
		}

		// Sanitize.
		$classes = array_map( 'esc_attr', $classes );

		// Set keys equal to vals.
		$classes = array_combine( $classes, $classes );

		/**
		 * Filters the header menu wrap classes.
		 *
		 * @param array $classes
		 */
		$classes = (array) apply_filters( 'wpex_header_menu_wrap_classes', $classes );

	}

	// Inner Classes.
	elseif ( 'inner' === $return ) {

		// Core.
		$classes[] = 'navigation';
		$classes[] = 'main-navigation';
		$classes[] = 'main-navigation-' . sanitize_html_class( $header_style );

		// Add the container class for specific header styles to center the menu items.
		if ( 'two' === $header_style || 'three' === $header_style || 'four' === $header_style ) {
			$classes[] = 'container';
			$classes[] = 'wpex-relative'; // !!important!!!
		}

		// Add 100% height class.
		if ( 'five' === $header_style ) {
			$classes[] = 'wpex-h-100';
		}

		// Add Clearfix.
		if ( $has_flex_header ) {
			if ( ! get_theme_mod( 'megamenu_stretch', true ) ) {
				$classes[] = 'wpex-relative';
			}
		} else {
			$classes[] = 'wpex-clr';
		}

		// Sanitize.
		$classes = array_map( 'esc_attr', $classes );

		// Set keys equal to vals.
		$classes = array_combine( $classes, $classes );

		/**
		 * Filters the header menu classes.
		 *
		 * @param array $classes
		 */
		$classes = (array) apply_filters( 'wpex_header_menu_classes', $classes );

	}

	if ( is_array( $classes ) ) {
		return implode( ' ', $classes );
	}

}

/**
 * Returns menu classes.
 */
function wpex_header_menu_ul_classes() {
	$classes         = [];
	$header_style    = Header::style();
	$dropdown_method = wpex_header_menu_dropdown_method();

	if ( 'dev' === $header_style ) {
		$classes[] = 'main-navigation-dev-ul';
	} else {
		$classes['main-navigation-ul'] = 'main-navigation-ul'; // !!! important - don't target dev !!!
		if ( ! wp_style_is( 'arm_bootstrap_all_css' ) && ! wp_style_is( 'bootstrap' ) && ! wp_style_is( 'toolset_bootstrap_4' ) ) {
			$classes['dropdown-menu'] = 'dropdown-menu';
		}
	}

	// Flex classes.
	if ( Header::has_flex_container() ) {
		$classes[] = 'wpex-flex';
		$classes[] = 'wpex-items-center';
		if ( ! Header_Menu::has_flush_dropdowns() && 'ten' === $header_style ) {
			$classes[] = 'wpex-flex-wrap';
		}
	}

	// Header specific classes.
	switch( $header_style ) {
		case 'five':
			$classes[] = 'wpex-flex';
			$classes[] = 'wpex-float-none';
			$classes[] = 'wpex-h-100';
			$classes[] = 'wpex-justify-center';
			$classes[] = 'wpex-items-center';
			break;
	}

	// Add dropdown classes.
	switch( $dropdown_method ) {
		case 'sfhover':
			$classes['sf-menu'] = 'sf-menu';
			break;
		case 'click':
			$classes[] = 'wpex-dropdown-menu';
			$classes[] = 'wpex-dropdown-menu--onclick';
			break;
		case 'hover':
		default;
			$classes[] = 'wpex-dropdown-menu';
			$classes[] = 'wpex-dropdown-menu--onhover';
			break;
	}

	// Dropdown animations.
	if ( get_theme_mod( 'menu_drodown_animate', false ) ) {
		switch( $dropdown_method ) {
			case 'sfhover':
				$classes[] = 'sf-menu--animate';
				break;
			default:
				$classes[] = 'wpex-dropdown-menu--animate';
				break;
		}
	}

	/**
	 * Filters the header menu ul classes.
	 *
	 * @param array $classes
	 */
	$classes = (array) apply_filters( 'wpex_header_menu_ul_classes', $classes );

	// Sanitize classes.
	$classes = array_unique( $classes );
	$classes = array_map( 'esc_attr', $classes );

	// Return class string.
	return implode( ' ', $classes );
}

/**
 * Custom menu walker.
 *
 * @todo move to inc/walkers/
 */
if ( ! class_exists( 'WPEX_Dropdown_Walker_Nav_Menu' ) ) {
	class WPEX_Dropdown_Walker_Nav_Menu extends Walker_Nav_Menu {
		function display_element( $element, &$children_elements, $max_depth, $depth, $args, &$output ) {
			$id_field        = $this->db_fields['id'];
			$header_style    = Header::style();
			$drop_arrow_type = get_theme_mod( 'menu_arrow' ) ?: 'angle';

			// Is megamenu.
			$is_megamenu = false;

			// Down Arrows.
			if ( ! empty( $children_elements[$element->$id_field] ) && ( $depth == 0 ) ) {

				$element->classes[] = 'dropdown';

				if ( get_theme_mod( 'menu_arrow_down' ) ) {

					if ( 'plus' === $drop_arrow_type ) {
						$arrow_icon = 'plus';
					} else {
						$arrow_dir = 'six' === $header_style ? 'right' : 'down';
						$arrow_icon = $drop_arrow_type . '-' . $arrow_dir;
					}

					$down_arrow = wpex_get_theme_icon_html( $arrow_icon, 'nav-arrow__icon' );

					/**
					 * Filters the down arrow icon html.
					 *
					 * @param string $down_arrow
					 * @param string $drop_arrow_type
					 */
					$down_arrow = apply_filters( 'wpex_header_menu_down_arrow_html', $down_arrow, $drop_arrow_type );

					if ( $down_arrow  ) {
						$element->title .= ' <span class="nav-arrow top-level">' . $down_arrow . '</span>';
					}

				}

			}

			// Mega menu arrows | @todo is it possible to ONLY add to megamenus?
			if ( $depth > 2 ) {
				$megamenu_icon = apply_filters( 'wpex_megamenu_sub_item_icon', 'angle-right');
				if ( $megamenu_icon ) {
					$element->title = '<span class="megamenu-sub-item-icon wpex-hidden wpex-mr-10">' . wpex_get_theme_icon_html( $megamenu_icon ) . '</span>' . $element->title;
				}
			}

			// Right/Left Arrows.
			if ( ! empty( $children_elements[$element->$id_field] ) && ( $depth > 0 ) ) {

				$element->classes[] = 'dropdown';

				if ( get_theme_mod( 'menu_arrow_side', true ) ) {

					if ( 'plus' === $drop_arrow_type ) {
						$arrow_icon = 'plus';
					} else {
						$arrow_dir = is_rtl() ? 'left' : 'right';
						$arrow_icon = $drop_arrow_type . '-' . $arrow_dir;
					}

					$side_arrow = wpex_get_theme_icon_html( $arrow_icon, 'nav-arrow__icon' );

					/**
					 * Filters the down arrow icon html.
					 *
					 * @param string $down_arrow
					 * @param string $drop_arrow_type
					 */
					$side_arrow = apply_filters( 'wpex_header_menu_down_side_html', $side_arrow, $drop_arrow_type );

					if ( $side_arrow  ) {
						$element->title .= ' <span class="nav-arrow second-level">' . $side_arrow . '</span>';
					}

				}

			}

			// Remove current menu item when using local-scroll class.
			if ( is_array( $element->classes )
				&& in_array( 'local-scroll', $element->classes )
				&& in_array( 'current-menu-item', $element->classes )
			) {
				$key = array_search( 'current-menu-item', $element->classes );
				unset( $element->classes[$key] );
			}

			// Define walker.
			Walker_Nav_Menu::display_element( $element, $children_elements, $max_depth, $depth, $args, $output );

		}
	}
}

/**
 * Checks for custom menus.
 */
function wpex_custom_menu() {
	$menu = get_post_meta( wpex_get_current_post_id(), 'wpex_custom_menu', true );

	if ( 'default' === $menu ) {
		$menu = '';
	}

	/**
	 * Filters the custom menu.
	 * Accepts a menu ID, slug, name, or object.
	 *
	 * @param int|string $menu
	 */
	$menu = apply_filters( 'wpex_custom_menu', $menu );

	return $menu;
}

/*-------------------------------------------------------------------------------*/
/* [ Dropdowns ]
/*-------------------------------------------------------------------------------*/

/**
 * Get header Menu dropdown style.
 */
function wpex_header_menu_dropdown_method() {
	$method = get_theme_mod( 'menu_dropdown_method' );

	/**
	 * Filters the header menu dropdown method.
	 *
	 * @param string $method.
	 */
	$style = apply_filters( 'wpex_header_menu_dropdown_method', $method );

	if ( ! $method ) {
		$method = 'sfhover';
	}

	return $method;
}

/**
 * Get header Menu dropdown style.
 */
function wpex_get_header_menu_dropdown_style() {
	$style   = get_theme_mod( 'menu_dropdown_style' );
	$post_id = wpex_get_current_post_id();

	if ( $post_id && Overlay_Header::is_enabled() && wpex_has_post_meta( 'wpex_overlay_header' ) ) {
		$meta = get_post_meta( $post_id, 'wpex_overlay_header_dropdown_style', true );
		if ( $meta ) {
			$style = $meta;
		}
	}

	/**
	 * Filters the header menu dropdown style.
	 *
	 * @param string $style.
	 */
	$style = apply_filters( 'wpex_header_menu_dropdown_style', $style );

	return $style;
}

/*-------------------------------------------------------------------------------*/
/* [ Drop Widgets ]
/*-------------------------------------------------------------------------------*/

/**
 * Get header Menu dropdown widget class.
 */
function wpex_get_header_drop_widget_class() {
	$dropdown_style = wpex_get_header_menu_dropdown_style();
	$has_border     = true;

	if ( 'black' === $dropdown_style ) {
		$has_border = get_theme_mod( 'menu_dropdown_top_border' );
	}

	$class = array(
		'header-drop-widget',
	);

	if ( $has_border ) {
		$class[] ='header-drop-widget--colored-top-border';
	}

	// Utility classes
	$class[] ='wpex-invisible';
	$class[] ='wpex-opacity-0';
	$class[] ='wpex-absolute';
	$class[] ='wpex-transition-all';
	$class[] ='wpex-duration-200';
	$class[] = 'wpex-translate-Z-0';
	$class[] = 'wpex-text-initial';
	$class[] = 'wpex-z-10000';

	if ( TotalTheme\Header\Vertical::is_enabled() ) {
		$class[] = 'wpex-top-0';
		$vertical_header_position = TotalTheme\Header\Vertical::position();
		$class[] = 'wpex-' . sanitize_html_class( $vertical_header_position ) . '-100'; // left/right 100%
	} else {
		$class[] = 'wpex-top-100';
		$class[] = 'wpex-right-0';
	}

	switch ( $dropdown_style ) {
		case 'black':
			$class[] = 'wpex-bg-black';
			$class[] = 'wpex-text-gray-600';
			break;
		default:
			$class[] = 'wpex-surface-1';
			$class[] = 'wpex-text-2';
			break;
	}

	/**
	 * Filters the header dropdown widget element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_get_header_drop_widget_class', $class );

	return $class;
}

/**
 * Header Menu dropdown widget class.
 */
function wpex_header_drop_widget_class() {
	$class = wpex_get_header_drop_widget_class();

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', $class ) ) . '"';
	}
}

/**
 * Check if header menu dropdown widgets should be added inline (insite the li element).
 */
function wpex_maybe_add_header_drop_widget_inline( $widget = '' ) {
	$check        = true;
	$header_style = Header::style();

	if ( 'one' === $header_style || Header::has_flex_container() ) {
		$check = false; // set items flush with the header bottom.
	}

	if ( function_exists( 'max_mega_menu_is_enabled' ) && max_mega_menu_is_enabled( 'main_menu' ) ) {
		$check = true;
	}

	/**
	 * Filters whether the header dropdown widget should display inline.
	 *
	 * @param bool $check
	 * @param string $widget
	 */
	$check = (bool) apply_filters( 'wpex_maybe_add_header_drop_widget_inline', $check, $widget );

	return $check;
}

/*-------------------------------------------------------------------------------*/
/* [ Mobile ]
/*-------------------------------------------------------------------------------*/

/**
 * Conditional check for alternative menu.
 *
 * @todo rename to wpex_has_header_menu_mobile_alt
 */
function wpex_has_mobile_menu_alt() {
	return (bool) apply_filters( 'wpex_has_mobile_menu_alt', has_nav_menu( 'mobile_menu_alt' ) );
}

/**
 * Return header mobile menu classes.
 *
 * @todo rename to wpex_mobile_menu_toggle_classes and deprecate
 */
function wpex_header_mobile_menu_classes() {
	$style        = wpex_header_menu_mobile_toggle_style();
	$header_style = Header::style();
	$flex_header  = Header::has_flex_container();

	$classes = array(
	//	'header-' . sanitize_html_class( $header_style ) . '-mobile-menu-toggle', // @todo should we add this?
		'wpex-mobile-menu-toggle',
		'show-at-mm-breakpoint',
		// Add flex styles to all toggles.
		'wpex-flex',
		'wpex-items-center',
	);

	switch ( $style ) {
		case 'icon_buttons':
			if ( ! $flex_header ) {
				$classes[] = 'wpex-absolute wpex-top-50 -wpex-translate-y-50 wpex-right-0';
			}
			break;
		case 'icon_buttons_under_logo':
			if ( $flex_header ) {
				$classes[] = 'wpex-mt-10';
			} else {
				$classes[] = 'wpex-mt-20';
			}
			$classes[] = 'wpex-justify-center';
			break;
		case 'navbar':
			$classes[] = 'wpex-bg-gray-A900';
			break;
		case 'fixed_top':
			$classes[] = 'wpex-fixed';
			$classes[] = 'wpex-z-9999';
			$classes[] = 'wpex-top-0';
			$classes[] = 'wpex-inset-x-0';
			$classes[] = 'wpex-surface-dark';
			//$classes[] = 'wpex-bg-gray-A900';
			break;
	}

	$styles_w_icons = array(
		'icon_buttons',
		'icon_buttons_under_logo',
		'centered_logo',
		'next_to_logo',
	);

	if ( $flex_header && in_array( $style, $styles_w_icons ) ) {
		$classes[] = 'wpex-h-100';
	}

	/**
	 * Filters the mobile menu toggle classes.
	 *
	 * @param array $classes
	 */
	$classes = (array) apply_filters( 'wpex_mobile_menu_toggle_class', $classes, $style );

	return esc_attr( implode( ' ', $classes ) );
}

/**
 * Returns classes for the header menu mobile toggle.
 */
function wpex_mobile_menu_toggle_class() {
	$classes = wpex_header_mobile_menu_classes(); // @todo rename

	if ( $classes ) {
		echo 'class="' . esc_attr( $classes ) . '"';
	}
}

/**
 * Return correct mobile menu toggle style for the header.
 */
function wpex_header_menu_mobile_toggle_style() {
	if ( 'disabled' === Mobile_Menu::style() ) {
		return false;
	}

	$toggle_style = get_theme_mod( 'mobile_menu_toggle_style' );

	if ( ( 'centered_logo' === $toggle_style || 'next_to_logo' === $toggle_style )
		&& ! Header::has_flex_container()
	) {
		$toggle_style = 'icon_buttons';
	}

	if ( ! $toggle_style ) {
		$toggle_style = 'icon_buttons';
	}

	/**
	 * Filters the header menu mobile toggle style.
	 *
	 * @param string $style
	 */
	$toggle_style = (string) apply_filters( 'wpex_mobile_menu_toggle_style', $toggle_style );

	return $toggle_style;
}

/**
 * Return sidr menu source.
 */
function wpex_sidr_menu_source( $deprecated = '' ) {
	$items = [];

	// Add mobile menu alternative if defined.
	if ( wpex_has_mobile_menu_alt() ) {
		$items['nav'] = '#mobile-menu-alternative';
	}

	// If mobile menu alternative is not defined add main navigation.
	else {
		$items['nav'] = '#site-navigation';
	}

	// Add search form.
	if ( get_theme_mod( 'mobile_menu_search', true ) ) {
		$items['search'] = '#mobile-menu-search';
	}

	/**
	 * Filters the mobile menu sidr source.
	 *
	 * @since 4.0
	 *
	 * @param array $items The array of elements (ids) to include in the sidebar mobile menu.
	 */
	$items = (array) apply_filters( 'wpex_mobile_menu_source', $items );

	return implode( ', ', $items );
}

/**
 * Return mobile toggle icon html.
 */
function wpex_get_mobile_menu_toggle_icon() {
	$mobile_menu_style = Mobile_Menu::style();
	$text              = wpex_get_translated_theme_mod( 'mobile_menu_icon_label' );
	$toggle_state      = get_theme_mod( 'mobile_menu_icon_toggle_state', true );

	/**
	 * Filters whether the hamburger icon should have a toggle state or not.
	 *
	 * @param bool $check
	 */
	$toggle_state = apply_filters( 'wpex_mobile_menu_icon_toggle_state', $toggle_state );

	$args = array(
		'toggle_state' => $toggle_state,
		'animate'      => get_theme_mod( 'mobile_menu_icon_animate', true ),
		'rounded'      => get_theme_mod( 'mobile_menu_icon_rounded', false ),
	);

	$icon = TotalTheme\Hamburger_Icon::render( $args );

	if ( $icon ) {
		$icon = '<span class="mobile-menu-toggle__icon wpex-flex">' . $icon . '</span>';
	}

	$el_class = 'mobile-menu-toggle';

	if ( $text ) {
		$el_class .= ' wpex-gap-10';

		$label_position = get_theme_mod( 'mobile_menu_icon_label_position' );

		if ( 'left' === $label_position ) {
			$el_class .= ' wpex-flex-row-reverse';
		}

		$text = '<span class="mobile-menu-toggle__label">' . esc_html( $text ) . '</span>';
	}

	// Begin output.
	$html = '<a href="#" class="' . esc_attr( $el_class ) . '" role="button" aria-label="' . esc_attr( wpex_get_aria_label( 'mobile_menu_toggle' ) ) .'" aria-expanded="false">';

		$button_text = $icon . $text;

		/**
		 * Filters the mobile menu toggle button text.
		 *
		 * @param string $text
		 */
		$html .= apply_filters( 'wpex_mobile_menu_open_button_text', $button_text );

	$html .= '</a>';

	$html = apply_filters( 'wpex_get_mobile_menu_toggle_icon', $html ); // @todo deprecate

	/**
	 * Filters the mobile menu toggle icon html.
	 *
	 * @since 5.0.7
	 *
	 * @param string $html The post thumbnail HTML.
	 */
	$html = apply_filters( 'wpex_mobile_menu_toggle_icon', $html );

	return $html;
}

/**
 * Return mobile menu extra icons.
 *
 * @todo rename menu area to mobile_toggle_icons.
 */
function wpex_mobile_menu_toggle_extra_icons() {
	$icons_escaped = '';

	if ( ( $locations = get_nav_menu_locations() ) && isset( $locations[ 'mobile_menu' ] ) ) {

		$menu = wp_get_nav_menu_object( $locations[ 'mobile_menu' ] );

		if ( ! empty( $menu ) ) {

			$menu_items = wp_get_nav_menu_items( $menu->term_id );

			if ( $menu_items ) {

				foreach ( $menu_items as $key => $menu_item ) :

					// Only add items if a correct font icon is added for the menu item label.
					if ( ! in_array( $menu_item->title, wpex_ticons_list() ) ) {
						continue;
					}

					$title = $menu_item->title;

					$reader_text = $title;
					$attr_title  = $menu_item->attr_title;
					$desc        = '';
					$link_icon   = wpex_get_theme_icon_html( $title );

					$a_class = array(
						'mobile-menu-extra-icons',
						'mobile-menu-' . sanitize_html_class( $title ),
						'wpex-inline-block',
						'wpex-no-underline',
					);

					// Get mobile menu toggle style.
					$toggle_style = wpex_header_menu_mobile_toggle_style();

					/// Add margin to icons
					$toggle_style_w_icons = array(
						'icon_buttons',
						'icon_buttons_under_logo',
						'centered_logo',
						'next_to_logo',
					);

					if ( in_array( $toggle_style, $toggle_style_w_icons ) ) {
						$a_class[] = 'wpex-mr-20'; // right margin
					} else {
						$a_class[] = 'wpex-ml-20'; // left margin
					}

					// @todo update to display count always.
					if ( (
							'shopping-cart' === $title
							|| 'shopping-bag' === $title
							|| 'shopping-basket' === $title
						)
						&& function_exists( 'wpex_mobile_menu_cart_count' )
					) {
						$link_icon = '<span class="wpex-relative wpex-inline-block">' . $link_icon . wpex_mobile_menu_cart_count() . '</span>';
					}

					if ( ! empty( $menu_item->classes ) && is_array( $menu_item->classes ) ) {
						$a_class = array_merge( $a_class, $menu_item->classes );
					}

					$link_attrs = array(
						'href'  => esc_url( $menu_item->url ),
						'class' => implode( ' ', array_map( 'esc_attr', $a_class ) ),
					);

					if ( '#search' === esc_url( $menu_item->url ) ) {
						$link_attrs['role'] = 'button';
						$link_attrs['aria-expanded'] = 'false';
					}

					if ( ! empty( $menu_item->description ) ) {
						$desc = '<span class="wpex-icon-label wpex-ml-5">' . esc_html( $menu_item->description ) . '</span>';
						$reader_text = '';
					} else {
						if ( ! empty( $attr_title ) ) {
							$link_attrs['title'] = esc_attr( $attr_title );
							$reader_text = $attr_title;
						}
						if ( $reader_text ) {
							$reader_text = '<span class="screen-reader-text">' . esc_html( $reader_text ) . '</span>';
						}
					}

					$inner_html = $link_icon . $reader_text . $desc;

					$icons_escaped .= wpex_parse_html( 'a', $link_attrs, $inner_html );

				endforeach; // End foreach.

			}

		}

	} // End menu check.

	$icons_escaped = apply_filters( 'wpex_get_mobile_menu_extra_icons', $icons_escaped ); // @todo deprecate legacy filter

	/**
	 * Filters the header menu mobile toggle icons html.
	 *
	 * @param string $html
	 */
	$icons_escaped = apply_filters( 'wpex_header_menu_mobile_toggle_icons', $icons_escaped );

	if ( $icons_escaped ) {
		echo '<div class="wpex-mobile-menu-toggle-extra-icons">' . $icons_escaped . '</div>';
	}
}