<?php
defined( 'ABSPATH' ) || exit;

/**
 * WooCommerce Template Shortcode.
 *
 * @package TotalThemeCore
 * @version 1.5
 */
if ( ! class_exists( 'VCEX_WooCommerce_Template_Shortcode' ) ) {

	class VCEX_WooCommerce_Template_Shortcode {

		/**
		 * Shortcode tag.
		 */
		const TAG = 'vcex_woocommerce_template';

		/**
		 * Main constructor.
		 */
		public function __construct() {
			add_shortcode( self::TAG, __CLASS__ . '::output' );

			if ( function_exists( 'vc_lean_map' ) ) {
				new \TotalThemeCore\Vcex\WPBakery\Map\WooCommerce_Template;
			}
		}

		/**
		 * Get shortcode title.
		 */
		public static function get_title() {
			return esc_html__( 'WooCommerce Template Part', 'total-theme-core' );
		}

		/**
		 * Get shortcode description.
		 */
		public static function get_description() {
			return esc_html__( 'Returns WooCommerce template parts', 'total-theme-core' );
		}

		/**
		 * Shortcode output => Get template file and display shortcode.
		 */
		public static function output( $atts, $content = null ) {
			if ( ! vcex_maybe_display_shortcode( self::TAG, $atts ) || ! function_exists( 'wc_get_product' ) ) {
				return;
			}

			$atts = vcex_shortcode_atts( self::TAG, $atts, get_class() );
			$template_choices = self::get_template_choices();

			if ( empty( $atts['part'] ) || ! array_key_exists( $atts['part'], $template_choices ) ) {
				return;
			}

			global $product;
			global $post;

			$html = '';
			$is_edit_mode = vcex_is_template_edit_mode();
			$unique_classname = vcex_element_unique_classname();

			$class = [
				'vcex-wc-template-part',
				'vcex-wc-template-part--' . self::parse_part_class( $atts['part'] ),
				'vcex-module',
				'woocommerce', // styles sometimes target this class
			];

			if ( ! empty( $atts['css_animation'] ) ) {
				$class[] = vcex_get_css_animation( $atts['css_animation'] );
			}

			if ( ! empty( $atts['text_align'] ) ) {
				$class[] = 'wpex-text-' . sanitize_html_class( $atts['text_align'] ); // !!! important!!!
			}

			if ( ! empty( $atts['css'] ) ) {
				$class[] = vcex_vc_shortcode_custom_css_class( $atts['css'] );
			}

			if ( ! empty( $atts['el_class'] ) ) {
				$class[] = vcex_get_extra_class( $atts['el_class'] );
			}

			$shortcode_style = vcex_inline_style( [
				'animation_delay'    => $atts['animation_delay'] ?? null,
				'animation_duration' => $atts['animation_duration'] ?? null,
			] );

			$typography_css = vcex_inline_style( [
				'color'                   => $atts['color'] ?? null,
				'--wpex-link-color'       => $atts['color'] ?? null,
				'--wpex-hover-link-color' => $atts['color'] ?? null,
				'--wpex-woo-price-color'  => $atts['color'] ?? null,
				'font_family'             => $atts['font_family'] ?? null,
				'font_size'               => $atts['font_size'] ?? null,
				'letter_spacing'          => $atts['letter_spacing'] ?? null,
				'font_weight'             => $atts['font_weight'] ?? null,
				'line_height'             => $atts['line_height'] ?? null,
				'text_transform'          => $atts['text_transform'] ?? null,
				'text_align'              => $atts['text_align'] ?? null,
			], false );

			$responsive_css = vcex_element_responsive_css( [
				'font_size' => $atts['font_size'] ?? null,
			], 'vcex-wc-template-part.'. $unique_classname . '> *' );

			if ( $typography_css || $responsive_css ) {
				$class[] = $unique_classname;
				$html .= '<style>';
					if ( $typography_css ) {
						$html .= '.vcex-wc-template-part.' . $unique_classname .' > *:not(.star-rating) {' . wp_strip_all_tags( $typography_css ) . '}'; // note: can't use esc_attr because it can break font families.
					}
					if ( $responsive_css ) {
						$html .= $responsive_css;
					}
				$html  .= '</style>';
			}

			$html .= '<div class="' . esc_attr( implode( ' ', array_filter( $class ) ) ) . '"' . $shortcode_style . '>';
				$product_data = null;

				if ( $is_edit_mode ) {

					$products = get_posts( [
						'numberposts' => 1,
						'post_type'   => 'product',
						'fields'      => 'ids',
					] );

					if ( isset( $products[0] ) ) {
						$temp_post    = $post;
						$post         = get_post( $products[0] );
						$product_data = wc_get_product( $products[0] );
					}

				} else {
					$product_data = get_post();
				}

				if ( $product_data ) {
					$product = is_object( $product_data ) && in_array( $product_data->post_type, [ 'product', 'product_variation' ], true ) ? wc_setup_product_data( $post ) : false;
					if ( $product ) {
						$template_html = self::get_template_part( $atts['part'], $product, $is_edit_mode );
						if ( $template_html ) {
							$html .= $template_html;
						} elseif ( $is_edit_mode ) {
							$html .= $template_choices[$atts['part']];
						}
						if ( $is_edit_mode && ! empty( $temp_post ) ) {
							$post = $temp_post;
						}
					}
					wc_setup_product_data( $post );
				}
			$html .= '</div>';
			return $html;
		}

		/**
		 * Returns template part.
		 */
		protected static function get_template_part( $part, $product, $is_edit_mode ) {
			if ( $is_edit_mode ) {
				switch ( $part ) {
					case 'single-product/tabs/description':
					case 'single-product/tabs/tabs':
						return '<div class="wpex-surface-3 wpex-text-2 wpex-p-15 wpex-text-center">' . esc_html__( 'This element is disabled in the frontend editor to prevent an endless loop.', 'total-theme-core' ) .'</div>';
						break;
					case 'single-product/product-image':
						if ( function_exists( 'wpex_get_placeholder_image' ) ) {
							return wpex_get_placeholder_image();
						}
						break;
				}
			}
			switch ( $part ) {
				// Entry
				case 'loop/title':
					if ( function_exists( 'woocommerce_template_loop_product_title' ) ) {
						ob_start();
							woocommerce_template_loop_product_title();
						return ob_get_clean();
					}
					break;
				case 'loop/thumbnail':
					if ( function_exists( 'woocommerce_template_loop_product_thumbnail' ) ) {
						ob_start();
							woocommerce_template_loop_product_thumbnail();
						return ob_get_clean();
					}
					break;
				case 'loop/add-to-cart':
					if ( function_exists( 'woocommerce_template_loop_add_to_cart' ) ) {
						ob_start();
							woocommerce_template_loop_add_to_cart();
						return ob_get_clean();
					}
					break;
				// Single
				case 'single-product/product-attributes':
					if ( function_exists( 'wc_display_product_attributes' ) ) {
						ob_start();
							wc_display_product_attributes( $product );
						return ob_get_clean();
					}
					break;
				case 'single-product/stock':
					if ( function_exists( 'wc_get_stock_html' ) ) {
						return wc_get_stock_html( $product );
					}
					break;
				case 'single-product/add-to-cart':
					if ( function_exists( 'woocommerce_template_single_add_to_cart' ) ) {
						ob_start();
							woocommerce_template_single_add_to_cart();
						return ob_get_clean();
					}
					break;
				case 'single-product/related':
					if ( function_exists( 'wpex_woocommerce_output_related_products' ) ) {
						ob_start();
							wpex_woocommerce_output_related_products();
						return ob_get_clean();
					} elseif ( function_exists( 'woocommerce_output_related_products' ) ) {
						ob_start();
							woocommerce_output_related_products();
						return ob_get_clean();
					}
					break;
				case 'single-product/up-sells':
					if ( function_exists( 'wpex_woocommerce_upsell_display' ) ) {
						ob_start();
							wpex_woocommerce_upsell_display();
						return ob_get_clean();
					} elseif ( function_exists( 'woocommerce_upsell_display' ) ) {
						ob_start();
							woocommerce_upsell_display();
						return ob_get_clean();
					}
					break;
				case 'single-product/reviews':
					if ( function_exists( 'comments_template' ) ) {
						ob_start();
							comments_template();
						return ob_get_clean();
					}
					break;
				default:
					if ( function_exists( 'wc_get_template' ) ) {
						ob_start();
							wc_get_template( $part . '.php' );
						return ob_get_clean();
					}
					break;
			}
		}

		/**
		 * Array of shortcode parameters.
		 */
		public static function get_params() {
			$params = array(
				array(
					'type' => 'vcex_select',
					'heading' => esc_html__( 'Template', 'total-theme-core' ),
					'param_name' => 'part',
				//	'admin_label' => true,
					'choices' => self::get_template_choices(),
					'editors' => [ 'wpbakery', 'elementor' ],
				),
				array(
					'type' => 'vcex_select',
					'heading' => esc_html__( 'Visibility', 'total-theme-core' ),
					'param_name' => 'visibility',
					'editors' => [ 'wpbakery', 'elementor' ],
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Extra class name', 'total-theme-core' ),
					'param_name' => 'el_class',
					'description' => vcex_shortcode_param_description( 'el_class' ),
					'editors' => [ 'wpbakery', 'elementor' ],
				),
				vcex_vc_map_add_css_animation(),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Animation Duration', 'total'),
					'param_name' => 'animation_duration',
					'description' => esc_html__( 'Enter your custom time in seconds (decimals allowed).', 'total'),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Animation Delay', 'total'),
					'param_name' => 'animation_delay',
					'description' => esc_html__( 'Enter your custom time in seconds (decimals allowed).', 'total'),
				),
				// Typography
				array(
					'type' => 'vcex_text_alignments',
					'heading' => esc_html__( 'Text Align', 'total-theme-core' ),
					'param_name' => 'text_align',
					'group' => esc_html__( 'Typography', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_colorpicker',
					'heading' => esc_html__( 'Color', 'total-theme-core' ),
					'param_name' => 'color',
					'group' => esc_html__( 'Typography', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_font_size',
					'heading' => esc_html__( 'Font Size', 'total-theme-core' ),
					'param_name' => 'font_size',
					'group' => esc_html__( 'Typography', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_font_family_select',
					'heading' => esc_html__( 'Font Family', 'total-theme-core' ),
					'param_name' => 'font_family',
					'group' => esc_html__( 'Typography', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_select',
					'heading' => esc_html__( 'Font Weight', 'total-theme-core' ),
					'param_name' => 'font_weight',
					'group' => esc_html__( 'Typography', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_select',
					'heading' => esc_html__( 'Text Transform', 'total-theme-core' ),
					'param_name' => 'text_transform',
					'group' => esc_html__( 'Typography', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_preset_textfield',
					'heading' => esc_html__( 'Line Height', 'total-theme-core' ),
					'param_name' => 'line_height',
					'choices' => 'line_height',
					'group' => esc_html__( 'Typography', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_preset_textfield',
					'heading' => esc_html__( 'Letter Spacing', 'total-theme-core' ),
					'param_name' => 'letter_spacing',
					'choices' => 'letter_spacing',
					'group' => esc_html__( 'Typography', 'total-theme-core' ),
				),
				// CSS
				array(
					'type' => 'css_editor',
					'heading' => esc_html__( 'CSS box', 'total-theme-core' ),
					'param_name' => 'css',
					'group' => esc_html__( 'CSS', 'total-theme-core' ),
				),
			);

			/**
			 * Filters the vcex_woocommerce_template element params.
			 *
			 * @param array $params
			 */
			$params = (array) apply_filters( 'vcex_shortcode_params', $params, self::TAG );

			return $params;
		}

		/**
		 * Returns classname based on part.
		 */
		protected static function parse_part_class( $part ) {
			if ( 'single-product/tabs/tabs' === $part ) {
				return 'single-product__tabs';
			}
			$class = str_replace( '/', '__', $part );
			return $class;
		}

		/**
		 * Returns array of allowed template choices.
		 */
		protected static function get_template_choices() {
			return [
				'' => esc_html__( 'Select a Template', 'total-theme-core' ),

				// Entry
				'loop/title' => esc_html__( 'Product Entry: Title', 'total-theme-core' ),
				'loop/thumbnail' => esc_html__( 'Product Entry: Thumbnail', 'total-theme-core' ),
				'loop/thumbnail/image-swap' => esc_html__( 'Product Entry: Thumbnail with Secondary Hover', 'total-theme-core' ),
				'loop/price' => esc_html__( 'Product Entry: Price', 'total-theme-core' ),
				'loop/rating' => esc_html__( 'Product Entry: Rating', 'total-theme-core' ),
				'loop/sale-flash' => esc_html__( 'Product Entry: Sale Flash', 'total-theme-core' ),
				'loop/add-to-cart' => esc_html__( 'Product Entry: Add to Cart', 'total-theme-core' ),

				// Single
				'single-product/product-image' => esc_html__( 'Single Product: Image', 'total-theme-core' ),
				'single-product/title' => esc_html__( 'Single Product: Title', 'total-theme-core' ),
				'single-product/sale-flash' => esc_html__( 'Single Product: Sale Flash', 'total-theme-core' ),
				'single-product/price' => esc_html__( 'Single Product: Price', 'total-theme-core' ),
				'single-product/stock' => esc_html__( 'Single Product: Stock', 'total-theme-core' ),
				'single-product/rating' => esc_html__( 'Single Product: Rating', 'total-theme-core' ),
				'single-product/short-description' => esc_html__( 'Single Product: Short Description', 'total-theme-core' ),
				'single-product/add-to-cart' => esc_html__( 'Single Product: Add to Cart', 'total-theme-core' ),
				'single-product/meta' => esc_html__( 'Single Product: Meta', 'total-theme-core' ),
				'single-product/product-attributes' => esc_html__( 'Single Product: Attributes', 'total-theme-core' ),
				'single-product/tabs/tabs' => esc_html__( 'Single Product: Tabs', 'total-theme-core' ),
				'single-product/tabs/description' => esc_html__( 'Single Product: Tab Description', 'total-theme-core' ),
				'single-product/tabs/additional-information' => esc_html__( 'Single Product: Tab Additional Information', 'total-theme-core' ),
				'single-product/reviews' => esc_html__( 'Single Product: Reviews', 'total-theme-core' ),
				'single-product/related' => esc_html__( 'Single Product: Related', 'total-theme-core' ),
				'single-product/up-sells' => esc_html__( 'Single Product: Up-Sells', 'total-theme-core' ),
			];
		}

	}

}
new VCEX_WooCommerce_Template_Shortcode;

if ( class_exists( 'WPBakeryShortCode' ) && ! class_exists( 'WPBakeryShortCode_Vcex_WooCommerce_Template' ) ) {
	class WPBakeryShortCode_Vcex_WooCommerce_Template extends WPBakeryShortCode {
		public function outputTitle( $title ) {
			$icon = $this->settings( 'icon' );
			return '<h4 class="wpb_element_title"><i class="vc_general vc_element-icon' . ( ! empty( $icon ) ? ' ' . $icon : '' ) . '" aria-hidden="true"></i><span class="vcex-heading-text">' . esc_html__( 'WooCommerce Template Part', 'total-theme-core' ) . '<span></span></span></h4>';
		}
	}
}