<?php
namespace TotalThemeCore\Vcex;

defined( 'ABSPATH' ) || exit;

/**
 * Generates the inline CSS for a shortcode.
 *
 * @package TotalThemeCore
 * @subpackage Vcex
 * @version 1.6.0
 */
final class Shortcode_CSS {

	/**
	 * Holds array of styles generated for the page.
	 */
	static protected $styles_list = [];

	/**
	 * Holds a counter of shortcodes and the items that have shown/rendered to generate the unique classname.
	 */
	static protected $shortcodes_counter = [];

	/**
	 * The shortcode unique classname.
	 */
	public $unique_classname;

	/**
	 * The shortcode class.
	 */
	protected $shortcode_class = '';

	/**
	 * The shortcode tag/name.
	 */
	protected $shotcode_tag = '';

	/**
	 * CSS array.
	 */
	protected $css_array = [];

	/**
	 * Checks if we should use important rules
	 */
	protected $use_important = false;

	/**
	 * Checks if we should target the wpex-theme body class.
	 */
	protected $target_body_class = false;

	/**
	 * Class Constructor.
	 */
	public function __construct( $shortcode_class = '', $shortcode_atts = [] ) {
		if ( ! is_callable( [ $shortcode_class, 'get_params' ] ) ) {
			return;
		}

		/*
		// this causes issues with items added inside the post content that only exist on some cards.
		if ( vcex_doing_loadmore() ) {
			return;
		}
		*/

		$this->shortcode_class = $shortcode_class;
		$this->shortcode_tag = $this->shortcode_class::TAG;

		$shortcode_params = $this->shortcode_class::get_params();

		if ( ! $shortcode_params || ! is_array( $shortcode_params ) ) {
			return;
		}

		$this->unique_classname = $this->get_unique_classname();
		$this->use_important = $this->important_check();

		$css_params = [];

		foreach ( $shortcode_params as $param_k => $param_v ) {
			if ( ! isset( $param_v['css'] )
				|| empty( $param_v['param_name'] )
				|| empty( $shortcode_atts[$param_v['param_name']] )
			) {
				continue;
			}

			$selector    = '';
			$value       = $shortcode_atts[$param_v['param_name']] ?? '';
			$media_query = '';

			if ( is_array( $param_v['css'] ) ) {
				$property = $param_v['css']['property'] ?? $param_v['param_name'];
				if ( ! empty( $param_v['css']['selector'] ) ) {
					$selectors = $param_v['css']['selector'];
					if ( is_array( $selectors ) ) {
						foreach ( $selectors as $k => $v ) {
							$selectors[$k] = $this->parse_selector( $v );
						}
						$selector = implode( ',', $selectors );
					} else {
						$selector = $this->parse_selector( $selectors );
					}
				}
				if ( ! empty( $param_v['css']['media_query'] ) ) {
					$media_query = $param_v['css']['media_query'];
				}
			} else {
				$property = ( true === $param_v['css'] ) ? $param_v['param_name'] : $param_v['css'];
			}

			if ( ! $selector ) {
				$selector = '.' . $this->get_shortcode_classname() . '.' . $this->unique_classname;
				if ( $this->target_body_class ) {
					$selector = '.wpex-theme ' . $selector;
				}
			}

			if ( ! $property || ! $value ) {
				continue;
			}

			$param_settings = [
				'selector'    => $selector,
				'property'    => $property,
				'val'         => $value,
				'media_query' => $media_query,
				'important'   => $param_v['css']['important'] ?? false,
			];

			$css_params[] = array_filter( $param_settings );

		} // end foreach

		if ( ! is_array( $css_params ) ) {
			return;
		}

		foreach ( $css_params as $css_param ) {
			$this->add_css_to_array( $css_param );
		}
	}

	/**
	 * Generates css for a specific param.
	 */
	protected function add_css_to_array( $param ) {
		$property = $this->parse_property( $param['property'] );
		if ( ! in_array( $property, [ 'margin', 'padding'] ) && false !== strpos( $param['val'], '|' ) ) {
			$this->parse_responsive_param( $param['selector'], $property, $param['val'] );
		} else {
			$css = vcex_inline_style( [
				$property => $param['val'],
			], false );
			if ( isset( $param['important'] ) && true === $param['important'] ) {
				$css = str_replace( ';', '!important;', $css );
			}
			if ( $css ) {
				$media_query = $param['media_query'] ?? 0;
				$this->add_to_css_array( $media_query, $param['selector'], $css );
			}
		}
	}

	/**
	 * Parses property name.
	 */
	protected function parse_property( $property = '' ) {
		if ( ! str_starts_with( $property, '--' ) ) {
			$property = str_replace( '-', '_', $property );
		}
		return $property;
 	}

 	/**
	 * Parses property name.
	 */
	protected function parse_responsive_param( $selector = '', $property = '', $value = '' ) {
		$parsed_value = vcex_parse_multi_attribute( $value );
		if ( ! is_array( $parsed_value ) ) {
			return;
		}
		foreach ( $parsed_value as $device => $device_value ) {
			$css = vcex_inline_style( [
				$property => $device_value,
			], false );
			if ( $css ) {
				$media_query = $this->get_media_query_from_device( $device ) ?: 0;
				$this->add_to_css_array( $media_query, $selector, $css );
			}
		}
	}

	/**
	 * Adds item to the css array.
	 */
	protected function add_to_css_array( $media_query, $selector, $css ) {
		$this->css_array[$media_query][$selector][] = $css;
	}

 	/**
	 * Loops through selectors and combines CSS.
	 */
	protected function parse_selectors_css( $selectors = [] ) {
		$css = '';
		foreach ( $selectors as $selector => $properties ) {
			$prop_string = implode( '', $properties );
			if ( $this->use_important ) {
				$prop_string = str_replace( ';', '!important;', $prop_string );
			}
			$css .= $selector . '{' . $prop_string . '}';
		}
		return $css;
	}

	/**
	 * Returns media query based on device prefix.
	 */
	protected function get_media_query_from_device( $device ) {
		$breakpoints = vcex_get_css_breakpoints();

		if ( ! empty( $breakpoints[$device] ) ) {
			return '@media (max-width:' . esc_attr( $breakpoints[$device] ) . ')';
		}
	}

	/**
	 * Returns CSS.
	 */
	public function get_css() {
		$css = '';
		if ( $this->css_array ) {
			foreach ( $this->css_array as $media_query => $selectors ) {
				$parsed_css = $this->parse_selectors_css( $selectors );
				if ( $media_query ) {
					$css .= $media_query . '{' . $parsed_css . '}';
				} else {
					$css .= $parsed_css;
				}
			}
		}
		return $css;
	}

	/**
	 * Returns style.
	 */
	public function render_style( $echo_style = true ) {
		$check_storage = $this->check_storage();
		$css = $this->get_css();
		if ( $check_storage ) {
			$css_to_store = str_replace( $this->unique_classname, 'vcex_unique_classname', $css );
			if ( $key = array_search( $css_to_store, self::$styles_list ) ) {
				$this->unique_classname = $key;
				return true;
			}
		}
		if ( $css ) {
			if ( $check_storage ) {
				self::$styles_list[$this->unique_classname] = $css_to_store;
			}
			/**
			 * Filters whether the class should render the style tag when the shortcodes is displayed.
			 *
			 * @param bool $render_style
			 */
			$render_style = (bool) apply_filters( 'vcex_shortcode_css_render_style', true );

			if ( $render_style ) {
				if ( $echo_style ) {
					echo '<style>' . $css . '</style>';
				} else {
					return $css;
				}
			}

			return true;
		}
	}

	/**
	 * Parses a selector to add the unique class id.
	 */
	protected function parse_selector( $selector ) {
		$target = '.' . $this->get_shortcode_classname() . '.' . $this->unique_classname;
		if ( 'self' === $selector || '{{WRAPPER}}' === $selector ) {
			return $target;
		}
		if ( $this->target_body_class ) {
			$target = '.wpex-theme ' . $target;
		}
		if ( false !== strpos( $selector, '{{WRAPPER}}' ) ) {
			return str_replace( '{{WRAPPER}}', $target, $selector );
		}
		if ( in_array( $selector, [ ':hover'] ) ) {
			return $target . $selector;
		} else {
			return $target . ' ' . $selector;
		}
	}

	/**
	 * Returns unique classname for element.
	 */
	protected function get_unique_classname() {
		// @todo add classname using element name and placement?
		// This could cause confusion though if people target with CSS then remove or move things?
		/*if ( isset( self::$shortcodes_counter[$this->shortcode_tag] ) ) {
			self::$shortcodes_counter[$this->shortcode_tag]++;
			$count = self::$shortcodes_counter[$this->shortcode_tag];
		} else {
			$count = 1;
			self::$shortcodes_counter[$this->shortcode_tag] = $count;
		}
		$classname = $this->get_shortcode_classname() . '-' . $count;
		*/
		$classname = vcex_element_unique_classname();
		return $classname;
	}

	/**
	 * Returns shortcode element classname.
	 */
	protected function get_shortcode_classname() {
		$classname = str_replace( '_', '-', $this->shortcode_tag );
		if ( $this->shortcode_has_wrapper( $this->shortcode_tag ) ) {
			$classname .= '-wrap';
		}
		return $classname;
	}

	/**
	 * Checks if a given shortcode has a wrapper.
	 */
	protected function shortcode_has_wrapper( $shortcode_tag ) {
		return in_array( $shortcode_tag, [ 'vcex_recent_news', 'vcex_countdown', 'vcex_image_ba' ] );
	}

	/**
	 * Checks if we should check for existing styles and apply existing ones to shortcodes with the same styling.
	 *
	 * @todo this should perhaps be enabled by default but would need the ability to disable via the admin or
	 * allow it to be enabled in the admin under "Optimizations".
	 *
	 * The issue with this function is if any theme function runs do_shortcode() on the post content without displaying it,
	 * before it's actually rendered, the CSS may not actually be inserted on the page.
	 */
	protected function check_storage() {
		$check = (bool) apply_filters( 'vcex_shortcode_css_store_styles', false );
		if ( $check && vcex_vc_is_inline() ) {
			$check = false;
		}
		return $check;
	}

	/**
	 * Checks if we should apply important attributes to all custom inline styles.
	 */
	protected function important_check() {
		return (bool) apply_filters( 'vcex_shortcode_css_use_important_rule', $this->use_important, $this->shortcode_tag );
	}

	/**
	 * Returns array of stored styles.
	 */
	public static function get_styles_list() {
		return self::$styles_list;
	}

}