<?php
namespace TotalThemeCore\Cards;

use WPEX_Meta_Factory;

defined( 'ABSPATH' ) || exit;

/**
 * Register meta options for theme cards.
 *
 * @package TotalThemeCore
 * @version 1.5.1
 */
class Meta {

	/**
	 * Our single class instance.
	 */
	private static $instance;

	/**
	 * Create or retrieve the class instance.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
			static::$instance->init_hooks();
		}

		return static::$instance;
	}

	/**
	 * Hook into actions and filters.
	 */
	public function init_hooks() {
		add_action( 'admin_init', [ $this, 'init' ] );
	}

	/**
	 * Initialize.
	 */
	public function init() {
		if ( ! self::is_enabled() || ! class_exists( 'WPEX_Meta_Factory' ) ) {
			return;
		}

		new WPEX_Meta_Factory( $this->card_metabox() );
	}

	/**
	 * Check if enabled.
	 */
	public static function is_enabled() {
		$check = true;

		/**
		 * Filters if the card metabox is enabled or not.
		 *
		 * @param bool $check
		 */
		$check = (bool) apply_filters( 'wpex_has_card_metabox', $check );

		return $check;
	}

	/**
	 * Card metabox settings.
	 */
	protected function card_metabox() {
		$post_types = [
			'post'         => 'post',
			'portfolio'    => 'portfolio',
			'staff'        => 'staff',
			'testimonials' => 'testimonials',
		];

		/**
		 * Filters the post types to add the card metabox to.
		 *
		 * @param array $post_types
		 */
		$post_types = (array) apply_filters( 'wpex_card_metabox_post_types', $post_types );

		$fields = [
			[
				'name' => esc_html__( 'Link Target', 'total-theme-core' ),
				'id'   => 'wpex_card_link_target',
				'type' => 'select',
				'choices' => [
					'' => esc_html__( 'Default', 'total-theme-core' ),
					'_blank' => esc_html__( 'New Tab', 'total-theme-core' ),
				],
			],
			[
				'name' => esc_html__( 'Link URL', 'total-theme-core' ),
				'id'   => 'wpex_card_url',
				'type' => 'text',
			],
			[
				'name' => esc_html__( 'Thumbnail', 'total-theme-core' ),
				'id'   => 'wpex_card_thumbnail',
				'type' => 'upload',
				'return' => 'id',
				'desc' => esc_html__( 'Select a custom thumbnail to override the featured image.', 'total-theme-core' ),
			],
			[
				'name' => esc_html__( 'Font Icon', 'total-theme-core' ),
				'id'   => 'wpex_card_icon',
				'type' => 'icon_select',
				'choices' => $this->choices_icons(),
				'desc' => esc_html__( 'Enter your custom Font Icon classname or click the button to select from the available theme icons.', 'total-theme-core' ),
			],
		];

		/**
		 * Filters the card metabox fields.
		 *
		 * @param array $fields
		 */
		$fields = (array) apply_filters( 'wpex_card_metabox_fields', $fields );

		if ( ! empty( $fields ) ) {
			return [
				'id'       => 'card',
				'title'    => esc_html__( 'Card Settings', 'total-theme-core' ),
				'screen'   => $post_types,
				'context'  => 'normal',
				'priority' => 'default',
				'fields'   => $fields
			];
		}
	}

	/**
	 * Icon choices.
	 *
	 * @todo update to use SVG's for the icon displays - will require a new "icon_select" meta type where we can pass on an array of SVG options.
	 *
	 * @todo update so the icon option only saves the icon name and not the full ticon ticon- unless it's an SVG then it can save as ticons/
	 */
	protected function choices_icons() {
		$icons_list = [];
		if ( function_exists( 'wpex_ticons_list' ) ) {
			$ticons = wpex_ticons_list();
			if ( $ticons && is_array( $ticons ) ) {
				foreach ( $ticons as $ticon ) {
					if ( false !== strpos( $ticon, 'custom/' ) ) {
						continue;
					}
					if ( 'none' === $ticon || '' === $ticon ) {
						$icons_list[] = esc_html__( 'Default', 'total' );
					} else {
						$icons_list['ticon ticon-' . trim( $ticon )] = $ticon;
					}
				}
			}
		}

		/**
		 * Filters the icons list for the card metabox.
		 *
		 * @param array $icons_list
		 */
		$icons_list = (array) apply_filters( 'wpex_card_meta_choices_icons', $icons_list );

		return $icons_list;
	}
}